/** 
 * \file EditorManager.cpp
 * \brief CEditorManager class is the boss. The manager. It creates 3D device (irrlicht of course),
 * creates GUI, and 3D environment so called Level Editor. Calls all the initialization functions and 
 * goes to the loop.
 * 
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include <irrlicht.h>
using namespace irr;
#include "EditorManager.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CEditorManager::CEditorManager()
{
	m_ID = 0;
	m_FS = NULL;
	m_WorkingDirectory = ".";
	lastFPS = -1;
}

/**
 * \brief Initialization function. Calls init of all sub classes (GUI, EdiLevel, etc..)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorManager::Init()
{
	//create irrlicht device
	CreateDevice();
	//create GUI
	m_pGuiManager = new CEditorGUI();
	m_pGuiManager->Init(this);
	//create "level 3D environment handling class": edilevel
	m_pEdiLevel = new CEditorLevel();
	m_pEdiLevel->Init(this);
	//we gonna use filesystem everuwhere so lets store the pointer
	m_FS = m_pDevice->getFileSystem();
	//and useful working dir
	m_WorkingDirectory = m_FS->getWorkingDirectory();

	m_ScriptEngine = new CScript();
	m_ScriptEngine->Init(m_pDevice,"media/Scripts/actions.script");
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CEditorManager::~CEditorManager()
{
}

/**
 * \brief Editor drawing loop. Draws GUI (2D) and current level (3D).
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorManager::Update()
{
	if(!m_pDriver->beginScene(true, true, SColor(255,100,101,140)))
	{
		// reset failed, propably because render target wasn't destroyed.
		if(m_pGuiManager->m_TextureToRenderOn && m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn) 
		{
			// remove rendertarget.
			m_pGuiManager->m_TextureToRenderOn->drop();
			m_pDriver->removeTexture(m_pGuiManager->m_TextureToRenderOn);
			m_pGuiManager->m_TextureToRenderOn = 0;

			m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn->drop();
			m_pDriver->removeTexture(m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn);
			m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn = 0;

			// this should call d3ddevice->reset.. now it should work because rendertarget was destroyed.
			m_pDriver->OnResize(m_pDriver->getScreenSize());
			m_pDriver->beginScene(true, true, 0);

			if(m_pGuiManager->m_wnd_properties)
				m_pGuiManager->m_wnd_properties->setRelativePosition(rect<s32>(m_pDriver->getScreenSize().Width-200,14,m_pDriver->getScreenSize().Width,480));

			// and let's create rendertarget again.
			m_pGuiManager->m_TextureToRenderOn = m_pDriver->createRenderTargetTexture(dimension2di(200,200));
			m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn = m_pDriver->createRenderTargetTexture(dimension2di(180,180));
		}
	}

	if (m_pGuiManager->m_wnd_ModelPreview && m_pGuiManager->m_TextureToRenderOn)
	{
		//set render target texture
		m_pDriver->setRenderTarget(m_pGuiManager->m_TextureToRenderOn, true, true, video::SColor(0,0,0,255));     

		//set photo camera as active camera
		m_pSceneManager->setActiveCamera(m_pGuiManager->m_PhotoCamera);

		//draw whole scene into render buffer
		m_pSceneManager->drawAll();                

		//set back old render target
		m_pDriver->setRenderTarget(0);      

		//set the level camera as active one
		m_pSceneManager->setActiveCamera(m_pEdiLevel->m_LevelCamera);
	}
	if (m_pGuiManager->m_wnd_containerContent && m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn)
	{
		//set render target texture
		m_pDriver->setRenderTarget(m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn, true, true, video::SColor(0,20,100,20));     

		//set photo camera as active camera
		m_pSceneManager->setActiveCamera(m_pGuiManager->m_PickCamera);

		//draw whole scene into render buffer
		m_pSceneManager->drawAll();                

		//set back old render target
		m_pDriver->setRenderTarget(0);      

		//set the level camera as active one
		m_pSceneManager->setActiveCamera(m_pEdiLevel->m_LevelCamera);
	}

	m_pSceneManager->drawAll();
	m_pEdiLevel->OnRender();
	m_pGUIEnvironment->drawAll();
	if(m_pGuiManager->m_wnd_ModelPreview)
	{
		m_pDriver->draw2DImage(m_pGuiManager->m_TextureToRenderOn, m_pGuiManager->m_wnd_ModelPreview->getAbsolutePosition().UpperLeftCorner + position2di(5, 25));
	}
	if (!m_pGuiManager->m_wnd_AddPickScriptAction && m_pGuiManager->m_wnd_containerContent)
	{
		m_pDriver->draw2DImage(m_pGuiManager->m_ContainerPickPreview_TextureToRenderOn, m_pGuiManager->m_wnd_containerContent->getAbsolutePosition().UpperLeftCorner + position2di(120, 40));
		m_pDriver->draw2DRectangle(SColor(255,50,155,0),rect<s32>(m_pGuiManager->m_wnd_containerContent->getAbsolutePosition().UpperLeftCorner + position2di(120, 225),m_pGuiManager->m_wnd_containerContent->getAbsolutePosition().UpperLeftCorner + position2di(120, 225) + position2di(64,64)));
		m_pDriver->draw2DImage(m_pGuiManager->m_slotTex, m_pGuiManager->m_wnd_containerContent->getAbsolutePosition().UpperLeftCorner + position2di(120, 225));
		m_pDriver->draw2DImage(m_pGuiManager->m_SelectedPickItemIcon_Texture, m_pGuiManager->m_wnd_containerContent->getAbsolutePosition().UpperLeftCorner + position2di(120, 225));
	}
	m_pDriver->endScene();

	// display frames per second in window title
	int fps = m_pDriver->getFPS();
	if (lastFPS != fps)
	{
		core::stringw str = L"Level Editor";
		str += " FPS:";
		str += fps;

		m_pDevice->setWindowCaption(str.c_str());
		lastFPS = fps;
	}

}

/**
 * \brief Creates the Irrlicht device and get pointers to the main subsytems
 * for later use, the Editor manager is the central interface point to the rendering engine
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorManager::CreateDevice()
{
	//create irrlicht device and send "this" as event receiver meaning this class should 
	//implement OnEvent function and handle its own events...
	m_pDevice = createDevice( video::EDT_DIRECT3D9, dimension2d<s32>(1000, 730), 16, false, false, false, this);
	m_pDevice->setResizeAble(true);
   	m_pDriver = m_pDevice->getVideoDriver();
    m_pSceneManager = m_pDevice->getSceneManager();
	m_pGUIEnvironment = m_pDevice->getGUIEnvironment();
	//change font to garamond
	//IGUIFont* font = m_pGUIEnvironment->getFont("media/Garamond12.xml");
	//m_pGUIEnvironment->getSkin()->setFont(font);
}

/**
 * \brief Returns a pointer to the Irrlicht Device subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
IrrlichtDevice* CEditorManager::getDevice()
{
	return m_pDevice;
}

/**
 * \brief Returns a pointer to the Irrlicht Driver subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
IVideoDriver* CEditorManager::getDriver()
{
	return m_pDriver;
}

/**
 * \brief Changes working directory to default one.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorManager::backToWorkingDirectory()
{
	m_FS->changeWorkingDirectoryTo(m_WorkingDirectory.c_str());
}

/**
 * \brief Returns a pointer to the Irrlicht SceneManager subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
ISceneManager* CEditorManager::getSceneMngr()
{
	return m_pSceneManager;
}

/**
 * \brief Returns a pointer to the Irrlicht GUI subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
IGUIEnvironment* CEditorManager::getGUIEnvironment()
{
	return m_pGUIEnvironment;
}

/**
 * \brief Main event handler derived from IEventHandler, this will be passed down 
 * to the current states keyboard handler. The state controls its own keyboard events
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorManager::OnEvent(const SEvent& event)
{
	if (!m_pDriver)
		return false;

	//Gui Handles its own events here
	bool gui_handles = m_pGuiManager->OnEvent(event);

	//3D level environment handles events here
	if (!gui_handles)
	{
		if (m_pEdiLevel->OnEvent(event))
			return true;
	}

	if (event.EventType == EET_KEY_INPUT_EVENT)
	{
		if(event.KeyInput.PressedDown){
			 if(event.KeyInput.Key==KEY_F2){
				//if(console->isVisible())
				return true;
			 }
			 else{
				//m_bKeys[event.KeyInput.Key] = event.KeyInput.PressedDown;
				// Pass input down to the specific game state keyboard handler
			}
		}

	}

	if (event.EventType == EET_MOUSE_INPUT_EVENT)
    {
        // Pass input down to the specific game state mouse handler
    }
	return false;
}


