/** 
 * \file EdGui.cpp
 * \brief GUI class creates menus and property windows and buttons and handles all the user clicks.
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "IGUITreeCtrl.h"
#include "CGUITreeCtrl.h"
#include "EDGui.h"
#include "../EditorManager.h"
#include "IGUIContainer.h"
#include "CGUIFileSaveDialog.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CEditorGUI::CEditorGUI()
{
	m_EditorManager = NULL;
	m_PropComboBox_Visible = NULL;
	m_PropEditBox_Name = NULL;
	m_PropEditBox_Pos = NULL;
	m_wnd_containerContent = NULL;
	m_wnd_AddPickScriptAction = NULL;
	m_wnd_AddTriggerScriptAction = NULL;
	m_wnd_AddNPCScriptAction = 0;
	m_DialogButton_Warning = 0;
	m_DialogButton_AddText = 0;
	m_bLinkDialogNodes = false;
	m_wnd_AddNPCNewText = 0;
	m_wnd_ModelPreview = 0;
	m_pPreviewPickableItem = 0;
	m_bFileSaveDialog = false;
	m_bFileOpenDialog = false;
	m_bElementPicking = false;
	m_bAddPickDirectly = false;
	m_TextureToRenderOn = 0;
	m_ContainerPickPreview_TextureToRenderOn = 0;
	m_SelectedPickItemIcon_Texture = 0;
	m_slotTex = 0;
	m_PhotoCamera = 0;
	m_PickCamera = 0;
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CEditorGUI::~CEditorGUI()
{
}

/**
 * This  helper function is recursive.
 * It walks through the tree of dialog nodes and sets them all to value 'enabled'.
 */
void PropagateEnabled(TreeNode* node, bool enabled)
{
	node->isEnabled = enabled;

	TreeNode* child = node->Child;
	while(child)
	{
		PropagateEnabled(child,enabled);
		child = child->Sibling;
	}
}

/**
 * \brief Writes script to edit box that is displaying script
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::WritePickScript(s32 index)
{
	if (index > -1)
	{
		if (index < (s32)m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.size())
		{
			stringw script = "<";
			script += m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str();
			script += " target=\"";
			script += m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.target.c_str();
			script += "\" >\n";
			for(u32 a=0; a<m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.size();a++)
			{
				script +="\t\t";
				script += m_EditorManager->getScriptEngine()->GetActionLine(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a]);
			}
			script += "</";
			script += m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str();
			script += ">";
			m_PickEditBox_Script->setText(script.c_str());
		}
	}
	else
	{
		m_PickEditBox_Script->setText(L"");
	}
}

/**
 * \brief Writes dialog script to edit box that is displaying script actions
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::WriteNPCDialogScriptAction()
{
	m_SelectedNode = m_DialogTree->GetSelected();
	if(m_SelectedNode)
	{
		if(m_SelectedNode != m_DialogTree->GetRootNode()->Child)
		{
			stringw script = "<Actions>\n";
			for(u32 a=0; a < m_SelectedNode->scriptAction.actions.size(); a++)
			{
				script +="\t";
				script += m_EditorManager->getScriptEngine()->GetActionLine(m_SelectedNode->scriptAction.actions[a]);
			}
			script += "</Actions>";
			m_DialogEditBox_ActionsText->setText(script.c_str());
		}
		else
		{
			m_DialogEditBox_ActionsText->setText(L"");
		}
	}
	else
	{
		m_DialogEditBox_ActionsText->setText(L"");
	}
}

/**
 * \brief Writes trigger script to edit box that is displaying script
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::WriteTriggerScript(s32 index)
{
	if (index > -1)
	{
		if (index < (s32)m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.size())
		{
			stringw script = "<";
			script += m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str();
			script += " state=\"";
			script += m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.state.c_str();
			script += "\" >\n";
			for(u32 a=0; a<m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.size();a++)
			{
				script +="\t\t";
				script += m_EditorManager->getScriptEngine()->GetActionLine(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a]);
			}
			script += "</";
			script += m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str();
			script += ">";
			m_PickEditBox_Script->setText(script.c_str());
		}
	}
	else
	{
		m_PickEditBox_Script->setText(L"");
	}
}

/**
 * \brief Save script to given file
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::SavePickScript(stringc file)
{
	if (file != stringc(""))
	{
		m_EditorManager->getDevice()->getFileSystem()->changeWorkingDirectoryTo("media/Scripts/Pickables");

		io::IXMLWriter* xml = m_EditorManager->getDevice()->getFileSystem()->createXMLWriter(file.c_str());

		if (xml)
		{
			xml->writeXMLHeader(); xml->writeLineBreak();
			if(m_PickCheckBox_Equipable->isChecked())
			{
				xml->writeElement(L"Equipable",true,L"value",L"true");
				xml->writeLineBreak();
			}
			if(m_PickCheckBox_Usable->isChecked())
			{
				xml->writeElement(L"Usable",true,L"value",L"true");
				xml->writeLineBreak();
			}
			xml->writeLineBreak();
			xml->writeElement(L"Script",false); xml->writeLineBreak();
			for (u32 index = 0; index < m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.size(); index++)
			{
				xml->writeElement(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str(),false,
					L"target", m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.target.c_str());
				xml->writeLineBreak();
				for ( u32 a = 0; a < m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.size(); a++)
				{
					//stringw temp = m_ListOfGameObjects[id]->GetPickableItem(index).c_str();
					xml->writeElement(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].name.c_str(),true,
						L"target",m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].target.c_str(),
						L"attribute",m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].attribute.c_str(),
						L"value", m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].value.c_str());
					xml->writeLineBreak();
				}
				xml->writeClosingTag(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str()); xml->writeLineBreak();
			}

			xml->writeClosingTag(L"Script"); xml->writeLineBreak();
			xml->drop(); // don't forget to delete the xml reader
		}
		m_EditorManager->backToWorkingDirectory();
	}
	else
	{
		//script file does not exist for this element, this should never happen
		stringw message  = "Script filename is missing!";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Abort!", message.c_str());
	}
}

void CEditorGUI::SaveTriggerScript(stringc file)
{
	if (file != stringc(""))
	{
		m_EditorManager->getDevice()->getFileSystem()->changeWorkingDirectoryTo("media/Scripts/Static");

		io::IXMLWriter* xml = m_EditorManager->getDevice()->getFileSystem()->createXMLWriter(file.c_str());

		if (xml)
		{
			xml->writeXMLHeader(); xml->writeLineBreak();

			xml->writeElement(L"State",true,L"value",m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->state.c_str());
			xml->writeLineBreak();
			xml->writeLineBreak();

			xml->writeElement(L"Script",false); xml->writeLineBreak();
			for (u32 index = 0; index < m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.size(); index++)
			{
				xml->writeElement(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str(),false,
					L"state", m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.state.c_str());
				xml->writeLineBreak();
				for ( u32 a = 0; a < m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.size(); a++)
				{
					//stringw temp = m_ListOfGameObjects[id]->GetPickableItem(index).c_str();
					xml->writeElement(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].name.c_str(),true,
						L"target",m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].target.c_str(),
						L"attribute",m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].attribute.c_str(),
						L"value", m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions[a].value.c_str());
					xml->writeLineBreak();
				}
				xml->writeClosingTag(m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].event.name.c_str()); xml->writeLineBreak();
			}

			xml->writeClosingTag(L"Script"); xml->writeLineBreak();
			xml->drop(); // don't forget to delete the xml reader
		}
		m_EditorManager->backToWorkingDirectory();
	}
	else
	{
		//script file does not exist for this element, this should never happen
		stringw message  = "Script filename is missing!";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Abort!", message.c_str());
	}
}

//check consistency of dialog node actions, if action is pointing to deleted node - remove it
void CEditorGUI::CheckDialogNodeActions(TreeNode* node, u32 deletedNodeID)
{
	if(node->hasScriptAction)
	{
		for (u32 index = 0; index < node->scriptAction.actions.size(); index++)
		{
			if(node->scriptAction.actions[index].name == stringw(L"SetDialogNodeEnabled"))
			{
				if(node->scriptAction.actions[index].target == stringw(deletedNodeID))
				{
					node->scriptAction.actions.erase(index);
					if(node->scriptAction.actions.size() == 0)
					{
						//if node has no more actions
						node->hasScriptAction = false;
					}
					break; //here we assume there are no two actions pointing at the same deleted node!!!
				}
			}
		}
	}
	if (node->Child)
	{
		CheckDialogNodeActions(node->Child, deletedNodeID);
	}
	if (node->Sibling)
	{
		CheckDialogNodeActions(node->Sibling, deletedNodeID);
	}
}

//Write dialog node attributes to XML like file (recursive)
void CEditorGUI::WriteElement(IXMLWriter* xml, TreeNode* node)
{
	xml->writeElement(L"Node",false); xml->writeLineBreak();
	stringw text_id = node->ID;
	if(node->isEnabled)
	{
		xml->writeElement(L"Enabled",true,L"value",L"true"); xml->writeLineBreak();
	}
	else
	{
		xml->writeElement(L"Enabled",true,L"value",L"false"); xml->writeLineBreak();
	}
	xml->writeElement(L"Id",true,L"value",text_id.c_str()); xml->writeLineBreak();
	if(node->Link)
	{
		stringw text_link = node->Link->ID;
		xml->writeElement(L"Link",true,L"value",text_link.c_str()); xml->writeLineBreak();
	}
	if(node->isLink)
	{
		xml->writeElement(L"IsLink",true,L"value",L"true"); xml->writeLineBreak();
	}
	if(node->Player)
	{
		xml->writeElement(L"Speaker",true,L"value",L"PLAYER"); xml->writeLineBreak();
	}
	else
	{
		stringw text_name = node->NPCName;
		xml->writeElement(L"Speaker",true,L"value",text_name.c_str()); xml->writeLineBreak();
	}
	xml->writeElement(L"Text",true,L"value",node->Text.c_str()); xml->writeLineBreak();
	//write actions
	if(node->scriptAction.actions.size() > 0)
	{
		xml->writeElement(L"Actions",false); xml->writeLineBreak();
		for ( u32 a = 0; a < node->scriptAction.actions.size(); a++)
		{
			xml->writeElement(node->scriptAction.actions[a].name.c_str(),true,
				L"target",node->scriptAction.actions[a].target.c_str(),
				L"attribute",node->scriptAction.actions[a].attribute.c_str(),
				L"value",node->scriptAction.actions[a].value.c_str());
			xml->writeLineBreak();
		}
		xml->writeClosingTag(L"Actions"); xml->writeLineBreak();
	}

	if (node->Child)
	{
		WriteElement(xml, node->Child); //continue populating child
	}

	xml->writeClosingTag(L"Node"); xml->writeLineBreak();

	if (node->Sibling)
	{
		WriteElement(xml, node->Sibling); //continue populating sibilings.
	}
}

//Saves NPC conversation and actions related to dialog nodes in one XML like file
void CEditorGUI::SaveDialog(stringc file)
{
	if (file != stringc(""))
	{
		m_EditorManager->getDevice()->getFileSystem()->changeWorkingDirectoryTo("media/Dialogs");

		IXMLWriter* xml = m_EditorManager->getDevice()->getFileSystem()->createXMLWriter(file.c_str());
		
		if (xml)
		{
			xml->writeXMLHeader(); xml->writeLineBreak();

			TreeNode* node = m_DialogTree->GetRootNode()->Child;

			//recursive function
			WriteElement(xml, node);
			
			xml->drop(); // don't forget to delete the xml reader
		}
		m_EditorManager->backToWorkingDirectory();
	}
	else
	{
		//script file does not exist for this element, this should never happen
		stringw message  = "Unexpected error: Dialog filename is missing!";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Abort!", message.c_str());
	}
}

/**
 * \brief Event handler - handles user clicks (mouse and keyboard).
 *
 * Handles menu items, if menu item was selected - appropriate function is called (save, open, new...)
 * Handles button clicks, tree node selection, keyboard entries etc.
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorGUI::OnEvent(const SEvent& event)
{
	if(event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		if ((event.MouseInput.Event == EMIE_LMOUSE_PRESSED_DOWN) || (event.MouseInput.Event == EMIE_LMOUSE_LEFT_UP))
		{
			position2di mouse_pos(event.MouseInput.X, event.MouseInput.Y);
			//if mouse click is within GUI windows, we declare this event handled, and do nothing
			//because GUI event will be fired moment later and we handle it below
			if (m_wnd_gameItems->getAbsolutePosition().isPointInside(mouse_pos) || 
				m_wnd_properties->getAbsolutePosition().isPointInside(mouse_pos))
			{
				return true;
			}
			if (m_wnd_containerContent || m_wnd_AddPickScriptAction || m_wnd_AddTriggerScriptAction || m_wnd_AddNPCScriptAction)
			{
				return true;
			}
		}
	}
	if(event.EventType == EET_KEY_INPUT_EVENT)
	{
		if (m_wnd_AddNPCNewText && m_DialogButton_AddText)
		{
			//add dialog node on enter!
			if ((!event.KeyInput.PressedDown) && (event.KeyInput.Key == KEY_RETURN))
			{
				//insert new node
				TreeNode* node = m_DialogTree->InsertNode(m_DialogEditBox_Text->getText(),m_SelectedNode,m_NumNodes+1);
				if(node)
				{
					m_NumNodes++;
					if(m_SelectedNode != m_DialogTree->GetRootNode())
					{
						node->Player = !m_SelectedNode->Player;
					}
					else
					{
						node->Player = false;
					}
					if(!node->Player)
					{
						node->NPCName = m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->name;
					}
				}
				else
				{
					//error adding new node!
				}
				m_wnd_AddNPCNewText->remove();
				m_DialogButton_Warning = 0;
				m_DialogButton_AddText = 0;
				m_wnd_AddNPCNewText = 0;
				m_EditorManager->getGUIEnvironment()->setFocus(m_DialogButton_AddNode);

				/*SEvent NewEvent;
				NewEvent.EventType = irr::EET_GUI_EVENT;
				NewEvent.GUIEvent.Caller = m_DialogButton_AddText;
				NewEvent.GUIEvent.EventType = (irr::gui::EGUI_EVENT_TYPE)(EGET_BUTTON_CLICKED);
				m_EditorManager->getGUIEnvironment()->getRootGUIElement()->OnEvent(NewEvent);*/
				
				return true;
			}
		}
	}
	//GUI event is special kind of event (not simple mouse or keyboard) generated by irrlicht engine
	//it simplifies event handling by telling us directly if button was clicked or window was moved 
	//or menu item was selected... etc.
	if (event.EventType == EET_GUI_EVENT)
		{
			s32 id = event.GUIEvent.Caller->getID();
			IGUIEnvironment* env = m_EditorManager->getGUIEnvironment();

			switch(event.GUIEvent.EventType)
			{
			case EGET_ELEMENT_HOVERED:
				{
					if((id > 6000) && (id < 6090))
					{
						m_bElementPicking = true;
					}
					/*else if( id==4000)
					{
						m_bElementPicking = true;
					}*/
				}
				break;
			case EGET_ELEMENT_LEFT:
				{
					if((id > 6000) && (id < 6090))
					{
						m_bElementPicking = false;
						//bits redundant to load model again, but it worksaround problem with map loading (scene delete)
						CGUITreeCtrl* tree = (CGUITreeCtrl*)event.GUIEvent.Caller;
						tree->m_SavedNode = 0;
					}
				}
				break;
			case EGET_TREEVIEW_MOUSEHOVERNODE:
				{
					//Its better to add icons by clicking button Add, instead on mouse click
					if (id == 6100) //Container Content Tree of Pickables
					{
						IGUITreeCtrl* tree = (IGUITreeCtrl*)event.GUIEvent.Caller;
						//SetIconAtHand(tree->GetSelected()->Parent->Text, tree->GetSelected()->Text);
						return true;
					}
					else if (m_wnd_ModelPreview && m_bElementPicking && (id > 6000) && (id < 6090))
					{
						CGUITreeCtrl* tree = (CGUITreeCtrl*)event.GUIEvent.Caller;
						SendModelToPhotoSession(tree->m_SavedNode->Parent->Text, tree->m_SavedNode->Text);
						return true;
					}
				}
				break;
			case EGET_TREEVIEW_MOUSECLICK_SELECTION:
				//this is tree event, node selection with mouse click.
				{
					//Its better to add icons by clicking button Add, instead on mouse click
					if (id == 6100) //Container Content Tree of Pickables
					{
						IGUITreeCtrl* tree = (IGUITreeCtrl*)event.GUIEvent.Caller;
						SetIconAtHand(tree->GetSelected()->Parent->Text, tree->GetSelected()->Text);
						return true;
					}
					else if ((id > 6000) && (id < 6090))
					{
						IGUITreeCtrl* tree = (IGUITreeCtrl*)event.GUIEvent.Caller;
						SetElementAtHand(tree->GetSelected()->Parent->Text, tree->GetSelected()->Text);
						return true;
					}
					else if (id == 714)
					{
						IGUITreeCtrl* tree = (IGUITreeCtrl*)event.GUIEvent.Caller;
						TreeNode* link = tree->GetSelected();
						m_DialogEditBox_SelectedNodeID->setText(stringw(link->ID).c_str());
						if (m_bLinkDialogNodes)
						{
							//This code will create link to selected node (from previously selected node)
							//new tree node is created to represent this link in green color
							if (link->Player)
							{
								//TODO: must select NPC text node! Try Again.
								return false;
							}
							if (link == tree->GetRootNode()->Child)
							{
								//TODO: cant link to RootNode! Try Again.
								return false;
							}
							m_SelectedNode->Link = link;
							link = new TreeNode;
							link->isLink = true;
							link->hasScriptAction = false;
							link->Child = 0;
							link->Parent = m_SelectedNode;
							link->Sibling = 0;
							link->Link = 0;
							link->ID = ++m_NumNodes;
							link->Player = false;
							link->NPCName = tree->GetSelected()->NPCName;
							link->Text = tree->GetSelected()->Text;

							if(!m_SelectedNode->Child)	// Childless parent is simple
							{
								m_SelectedNode->Child = link;
							}
							else
							{
								TreeNode *curnode = m_SelectedNode->Child;
								while(curnode->Sibling)
									curnode = curnode->Sibling;
								curnode->Sibling = link;
								// Find last sibling of parent and include new sibling
							}
							m_bLinkDialogNodes = false;
						}
						//This line sets m_SelectedNode to new one, and writes its actions to the window
						WriteNPCDialogScriptAction();
						m_DialogEditBox_ChangeText->setText(stringw(link->Text).c_str());
					}
				}
				break;
			case EGET_ELEMENT_CLOSED:
				{
					if(event.GUIEvent.Caller == m_wnd_containerContent)
						m_wnd_containerContent = 0;
					if(event.GUIEvent.Caller == m_wnd_AddPickScriptAction)
						m_wnd_AddPickScriptAction = NULL;
					if(event.GUIEvent.Caller == m_wnd_AddTriggerScriptAction)
						m_wnd_AddTriggerScriptAction = NULL;
					if(event.GUIEvent.Caller == m_wnd_ModelPreview)
						m_wnd_ModelPreview = 0;
					if(event.GUIEvent.Caller == m_wnd_AddNPCScriptAction)
						m_wnd_AddNPCScriptAction = 0;
					if(event.GUIEvent.Caller == m_wnd_AddNPCNewText)
						m_wnd_AddNPCNewText = 0;
				}
				break;
			case EGET_FILE_CHOOSE_DIALOG_CANCELLED:
				{
					m_bFileOpenDialog = false;
					m_bFileSaveDialog = false;
				}
				break;
			case EGET_FILE_SELECTED:
				{
					stringc mapname;
					if (m_bFileOpenDialog)
					{
						// load the map file, selected in the file open dialog
						m_EditorManager->getEdiLevel()->SetLoadingDir(m_EditorManager->getFS()->getWorkingDirectory());
						m_EditorManager->backToWorkingDirectory();
						IGUIFileOpenDialog* dialog = (IGUIFileOpenDialog*)event.GUIEvent.Caller;
						mapname = dialog->getFileName();
						m_EditorManager->getEdiLevel()->OnLoadMap(mapname);
						//recreate cameras
						m_PhotoCamera = m_EditorManager->getSceneMngr()->addCameraSceneNode(0, 
							core::vector3df(0,-10000,0),
							core::vector3df(0,-10000,-100));
						m_PickCamera = m_EditorManager->getSceneMngr()->addCameraSceneNode(0, 
							core::vector3df(0,-10000,0),
							core::vector3df(0,-10000,100));
						//m_EditorManager->getEdiLevel()->CreateCamera(vector3df(100,300,100));
						m_bFileOpenDialog = false;
					}
                    else if (m_bFileSaveDialog)
					{
						//save the scene to selected map file
						IGUIFileOpenDialog* dialog = (IGUIFileOpenDialog*)event.GUIEvent.Caller;
						mapname = dialog->getFileName();
						m_EditorManager->getEdiLevel()->SetMapName(mapname);
						//we don't save our static preview cameras to the scene, so it is necessary to remove them
						m_PickCamera->remove();
						m_PhotoCamera->remove();
						m_EditorManager->getEdiLevel()->RemovePhotoSessionModel();
						//save map
						m_EditorManager->getSceneMngr()->saveScene(mapname.c_str(),m_EditorManager->getEdiLevel());
						//recreate cameras
						m_PhotoCamera = m_EditorManager->getSceneMngr()->addCameraSceneNode(0, 
							core::vector3df(0,-10000,0),
							core::vector3df(0,-10000,-100));
						m_PickCamera = m_EditorManager->getSceneMngr()->addCameraSceneNode(0, 
							core::vector3df(0,-10000,0),
							core::vector3df(0,-10000,100));
						m_EditorManager->backToWorkingDirectory();
					}
				}
				break;
			case EGET_MENU_ITEM_SELECTED:
				{
					// a menu item was clicked

					IGUIContextMenu* menu = (IGUIContextMenu*)event.GUIEvent.Caller;
					s32 id = menu->getItemCommandId(menu->getSelectedItem());
					if (id == 102) //Save Map
					{
						m_bFileSaveDialog = true;
						m_EditorManager->getDevice()->getFileSystem()->changeWorkingDirectoryTo("maps");
						CGUIFileSaveDialog* dialog = new CGUIFileSaveDialog(L"Save Map As", m_EditorManager->getGUIEnvironment(),
							m_EditorManager->getGUIEnvironment()->getRootGUIElement(),-1);
						dialog->drop();
					}
					else if (id == 101) //Load Map
					{
						m_bFileOpenDialog = true;
						m_EditorManager->getDevice()->getFileSystem()->changeWorkingDirectoryTo("maps");
						m_EditorManager->getGUIEnvironment()->addFileOpenDialog(L"Please select map to open");
					}
					else if (id == 500) //Help->About
					{
						stringw message  = "Level Editor v0.2 \nPetar Bajic www.mystic-peanut.com\n\n";
								message += "User Manual:\n";
								message += "Pick objects from left floating window and place them on the map. \n";
								message += "Move selected object by clicking on it and dragging it around (left mouse press and hold)\n";
								message += "Rotate selected object by pressing SHIFT + left mouse press and hold\n";
								message += "Scale selected object by pressing SHIFT + mouse scroll\n";
								message += "Rotate map with right mouse press and hold, pan with SHIFT + right mouse press and hold\n\n";
								message += "You can declare object a container in Properties floating window and add content to it.\n";
						m_EditorManager->getGUIEnvironment()->addMessageBox(L"About", message.c_str());
					}
				}
				break;
			case EGET_LISTBOX_CHANGED:
				{
					//PickScript Event ListBox changed
					if(m_wnd_AddPickScriptAction && (m_PickListBox_ExistingEvents->getID() == event.GUIEvent.Caller->getID()))
					{
						s32 index = m_PickListBox_ExistingEvents->getSelected();
						WritePickScript(index);
					}
					else if (m_wnd_AddTriggerScriptAction && (m_TriggerListBox_States->getID() == event.GUIEvent.Caller->getID()))
					{
						s32 index = m_TriggerListBox_States->getSelected();
						WriteTriggerScript(index);
					}
				}
				break;
			case EGET_CHECKBOX_CHANGED:
				{
					if(m_wnd_containerContent && (m_PickCheckBox_DirectPick->getID() == event.GUIEvent.Caller->getID()))
					{
						if (m_PickCheckBox_DirectPick->isChecked())
						{
							m_bAddPickDirectly = true;
							m_pContainer_AddPickItemToContainer->setEnabled(false);
						}
						else
						{
							m_bAddPickDirectly = false;
							m_pContainer_AddPickItemToContainer->setEnabled(true);
						}
					}
					else if(m_wnd_AddPickScriptAction && (m_PickCheckBox_Usable->getID() == event.GUIEvent.Caller->getID()))
					{
						if (m_PickCheckBox_Usable->isChecked())
						{
							m_PickComboBox_Events->addItem(L"OnUse");
							m_PickComboBox_Events->addItem(L"OnUseAgainst");
						}
						else
						{
							//removal is painful, but necessary (?)
							for (u32 index = 0; index < m_PickComboBox_Events->getItemCount(); index++)
							{
								if ((stringw(m_PickComboBox_Events->getItem(index)) == stringw("OnUse"))||
									(stringw(m_PickComboBox_Events->getItem(index)) == stringw("OnUseAgainst")))
								{
									m_PickComboBox_Events->removeItem(index);
									index--;//item removed, next one takes his place at his index
								}
							}
						}
					}
					else if(m_wnd_AddPickScriptAction && (m_PickCheckBox_Equipable->getID() == event.GUIEvent.Caller->getID()))
					{
						if (m_PickCheckBox_Equipable->isChecked())
						{
							m_PickComboBox_Events->addItem(L"OnEquip");
							m_PickComboBox_Events->addItem(L"OnUnequip");
						}
						else
						{
							//removal is painful, but necessary (?)
							for (u32 index = 0; index < m_PickComboBox_Events->getItemCount(); index++)
							{
								if ((stringw(m_PickComboBox_Events->getItem(index)) == stringw("OnEquip"))||
									(stringw(m_PickComboBox_Events->getItem(index)) == stringw("OnUnequip")))
								{
									m_PickComboBox_Events->removeItem(index);
									index--;//item removed size of list is reduced
								}
							}
						}
					}
				}
				break;
			case EGET_COMBO_BOX_CHANGED:
				{
					//container combo changed - enable button and store new object setting
					if(m_PropComboBox_Container->getID() == event.GUIEvent.Caller->getID())
					{
						if(m_PropComboBox_Container->getSelected()==0)
						{
							m_ContainerButton->setEnabled(true);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isContainer = true;
							}
						}
						else
						{
							m_ContainerButton->setEnabled(false);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isContainer = false;
							}
						}
					}
					//pickable combo changed
					else if(m_PropComboBox_Pickable->getID() == event.GUIEvent.Caller->getID())
					{
						if(m_PropComboBox_Pickable->getSelected()==0)
						{
							m_PickScriptButton->setEnabled(true);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isPickable = true;
							}
						}
						else
						{
							m_PickScriptButton->setEnabled(false);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isPickable = false;
							}
						}
					}
					//Trigger combo changed
					else if(m_PropComboBox_Trigger->getID() == event.GUIEvent.Caller->getID())
					{
						if(m_PropComboBox_Trigger->getSelected()==0)
						{
							m_TriggerScriptButton->setEnabled(true);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isTrigger = true;
							}
						}
						else
						{
							m_TriggerScriptButton->setEnabled(false);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isTrigger = false;
							}
						}
					}
					else if(m_PropComboBox_NPC->getID() == event.GUIEvent.Caller->getID())
					{
						if(m_PropComboBox_NPC->getSelected()==0)
						{
							m_NPCScriptButton->setEnabled(true);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isNPC = true;
							}
						}
						else
						{
							m_NPCScriptButton->setEnabled(false);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isNPC = false;
							}
						}
					}
					else if(m_PropComboBox_Monster->getID() == event.GUIEvent.Caller->getID())
					{
						if(m_PropComboBox_Monster->getSelected()==0)
						{
							m_MonsterScriptButton->setEnabled(true);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isMonster = true;
							}
						}
						else
						{
							m_MonsterScriptButton->setEnabled(false);
							if (m_EditorManager->getEdiLevel()->m_SelectedGameObject)
							{
								m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID()]->isMonster = false;
							}
						}
					}
					else if(m_wnd_AddPickScriptAction && (m_PickComboBox_ActionTarget->getID() == event.GUIEvent.Caller->getID()))
					{
						if(m_PickComboBox_ActionTarget->getSelected() == 3) //id
						{
							m_PickEditBox_ActionTargetID->setEnabled(true);
						}
						else
						{
							m_PickEditBox_ActionTargetID->setEnabled(false);
							m_PickEditBox_ActionTargetID->setText(L"");
						}
					}
					else if(m_wnd_AddNPCScriptAction && (m_DialogComboBox_ActionTarget->getID() == event.GUIEvent.Caller->getID()))
					{
						if(m_DialogComboBox_ActionTarget->getSelected() == 3) //id
						{
							m_DialogEditBox_ActionTargetID->setEnabled(true);
						}
						else
						{
							m_DialogEditBox_ActionTargetID->setEnabled(false);
							m_DialogEditBox_ActionTargetID->setText(L"");
						}
					}
				}
				break;
			case EGET_BUTTON_CLICKED:
				{
					//Button clicked: AddContainerContent
					if(!m_wnd_containerContent && !m_wnd_AddPickScriptAction && 
						m_ContainerButton->getID() == event.GUIEvent.Caller->getID())
					{
						//if there is selected object with container value set to true
						if(m_PropComboBox_Container->getSelected()==0)
						{
							if(m_EditorManager->getEdiLevel()->SingleObjectSelected())
							{
								m_SelectedObjectID = m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID();
								m_wnd_containerContent = MakeContainerContentWindow(m_EditorManager,env,m_EditorManager->getDevice(),m_SelectedObjectID);
							}
							else
							{
								//message: "select object you wish to fill with content"
								stringw message = "First select container object!";
								m_EditorManager->getGUIEnvironment()->addMessageBox(L"Error", message.c_str());
							}
						}
						return true;
					}
					//Container button OK!
					else if(m_wnd_containerContent && (m_pContainer_Ok->getID() == event.GUIEvent.Caller->getID()))
					{
						//Save Container Content and close the window
						IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(1357);

						CPickableItem* item;
						m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->ClearPickableItems();
						for ( u32 index = 0; index < container->GetNumItems(); index++)
						{
							item = container->GetItem(index);
							if (item != NULL)
							{
								if ((m_SelectedObjectID >= 0) && (m_SelectedObjectID < (s32)m_EditorManager->getEdiLevel()->m_ListOfGameObjects.size()))
								{
									m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->AddPickableItem(item->getName().c_str());
								}
							}
						}
						m_wnd_containerContent->remove();
						m_wnd_containerContent = NULL;
						return true;
					}
					else if(m_wnd_containerContent && (m_pContainer_AddPickItemToContainer->getID() == event.GUIEvent.Caller->getID()))
					{
						if(m_pPreviewPickableItem)
						{
							IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(1357);
							container->InsertItem(m_pPreviewPickableItem);
						}
					}
					else if (m_wnd_containerContent && !m_wnd_AddPickScriptAction && (m_pContainer_EditPickItemScript->getID() == event.GUIEvent.Caller->getID()))
					{
						//edit script of preview item
						m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.clear();
						if(m_pPreviewPickableItem)
						{
							m_wnd_AddPickScriptAction = MakePickScriptWindow(m_EditorManager,env,m_EditorManager->getDevice(),-1, m_pPreviewPickableItem->m_Script);
						}
					}
					//Button clicked: AddPickScriptAction 
					else if (!m_wnd_containerContent && !m_wnd_AddPickScriptAction && 
						m_PickScriptButton->getID() == event.GUIEvent.Caller->getID())
					{
						m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.clear();
						if(m_EditorManager->getEdiLevel()->SingleObjectSelected())
						{
							m_SelectedObjectID = m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID();
							if ((m_SelectedObjectID >= 0) && (m_SelectedObjectID < (s32)m_EditorManager->getEdiLevel()->m_ListOfGameObjects.size()))
							{
								//if script...
								m_wnd_AddPickScriptAction = MakePickScriptWindow(m_EditorManager,env,m_EditorManager->getDevice(),m_SelectedObjectID, m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->script);
							}
						}
					}
					//Button clicked: Add new pick script action
					else if(m_wnd_AddPickScriptAction && (m_PickButton_AddAction->getID() == event.GUIEvent.Caller->getID()))
					{
						s32 index = m_PickListBox_ExistingEvents->getSelected();
						TAction action;
						action.name = m_PickComboBox_Actions->getText();
						if(action.name != stringw(L"none"))
						{
							if(index>-1)
							{
								action.attribute = m_PickEditBox_ActionAttribute->getText();
								action.value = m_PickEditBox_ActionValue->getText();
								action.target = m_PickComboBox_ActionTarget->getText();
								if (action.target == stringw(L"id"))
								{
									if(stringw(m_PickEditBox_ActionTargetID->getText()) != stringw(L""))
									{
										action.target = m_PickEditBox_ActionTargetID->getText();
										m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.push_back(action);
										WritePickScript(index);
									}
									else
									{
										//TODO: message: ENTER ID Value for target of this action
									}
								}
								else
								{
									m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.push_back(action);
									WritePickScript(index);
								}
							}
							else
							{
								//TODO message: select event first!
							}
						}
						else
						{
							//TODO: message: "select action first!"
						}
					}
					//Button clicked: Add new pick script event
					else if(m_wnd_AddPickScriptAction && (m_PickButton_AddEvent->getID() == event.GUIEvent.Caller->getID()))
					{
						stringw event = m_PickComboBox_Events->getText();
						if(event != stringw(L"none"))
						{
							bool alreadyExists = false;
							for (u32 i=0; i<m_PickListBox_ExistingEvents->getItemCount(); i++)
							{
								if(stringw(m_PickListBox_ExistingEvents->getListItem(i)) == event)
								{
									alreadyExists = true;
								}
							}
							if(!alreadyExists)
							{
								m_PickListBox_ExistingEvents->addItem(event.c_str());
								m_PickListBox_ExistingEvents->setSelected(m_PickListBox_ExistingEvents->getItemCount()-1);
								TScriptAction scriptAction;
								scriptAction.event.name = event;
								scriptAction.event.target = m_PickComboBox_EventTarget->getText();
								m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.push_back(scriptAction);
								WritePickScript(m_PickListBox_ExistingEvents->getItemCount()-1);
							}
						}
						else
						{
							//TODO: msg: "Pick event first"
						}
					}
					//Button clicked: Remove pick script event
					else if(m_wnd_AddPickScriptAction && (m_PickButton_RemoveEvent->getID() == event.GUIEvent.Caller->getID()))
					{
						s32 index = m_PickListBox_ExistingEvents->getSelected();
						if(index > -1)
						{
							m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.erase(index);
							m_PickListBox_ExistingEvents->removeItem(index);
							u32 b = m_PickListBox_ExistingEvents->getItemCount();
							if(b > (u32)index)
							{
								m_PickListBox_ExistingEvents->setSelected(index);
								WritePickScript(index);
							}
							else
							{
								m_PickListBox_ExistingEvents->setSelected(index-1);
								WritePickScript(index-1);
							}
						}
						else
						{
							//TODO: msg: "Pick event first"
						}
					}
					//Button clicked: Save pick script
					else if(m_wnd_AddPickScriptAction && (m_PickButton_Save->getID() == event.GUIEvent.Caller->getID()))
					{
						SavePickScript(m_PickScript.c_str());
						m_wnd_AddPickScriptAction->remove();
						m_wnd_AddPickScriptAction = NULL;
					}
					//Button clicked: Cancel Pick Script
					else if(m_wnd_AddPickScriptAction && (m_PickButton_Cancel->getID() == event.GUIEvent.Caller->getID()))
					{
						m_wnd_AddPickScriptAction->remove();
						m_wnd_AddPickScriptAction = NULL;
					}
					//Button clicked: AddTriggerScriptAction MakeTriggerScriptWindow
					else if (m_TriggerScriptButton->getID() == event.GUIEvent.Caller->getID())
					{
						m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.clear();
						if(m_EditorManager->getEdiLevel()->SingleObjectSelected())
						{
							m_SelectedObjectID = m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID();
							if ((m_SelectedObjectID >= 0) && (m_SelectedObjectID < (s32)m_EditorManager->getEdiLevel()->m_ListOfGameObjects.size()))
							{
								m_wnd_AddTriggerScriptAction = MakeTriggerScriptWindow(m_EditorManager,env,m_EditorManager->getDevice(),m_SelectedObjectID, m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->script);
							}
						}
					}
					else if (m_wnd_AddTriggerScriptAction && (m_TriggerButton_AddState->getID() == event.GUIEvent.Caller->getID()))
					{
						//add new state
						if(stringw(m_TriggerEditBox_State->getText()) != stringw(L""))
						{
							bool alreadyExists = false;
							for (u32 i=0; i<m_TriggerListBox_States->getItemCount(); i++)
							{
								if(stringw(m_TriggerListBox_States->getListItem(i)) == stringw(m_TriggerEditBox_State->getText()))
								{
									alreadyExists = true;
								}
							}
							if(!alreadyExists)
							{
								m_TriggerListBox_States->addItem(m_TriggerEditBox_State->getText());
								m_TriggerListBox_States->setSelected(m_TriggerListBox_States->getItemCount()-1);
								TScriptAction scriptAction;
								scriptAction.event.name = L"OnClick";
								scriptAction.event.state = m_TriggerEditBox_State->getText();
								scriptAction.event.target = L"";
								m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.push_back(scriptAction);
								WriteTriggerScript(m_TriggerListBox_States->getItemCount()-1);
								m_TriggerEditBox_State->setText(L"");
							}
						}
						else
						{
							//TODO: msg: "First enter state in field above"
						}
					}
					else if (m_wnd_AddTriggerScriptAction && (m_TriggerButton_SetState->getID() == event.GUIEvent.Caller->getID()))
					{
						//add new state
						s32 index = m_TriggerListBox_States->getSelected();
						if(index > -1)
						{
							m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->state = m_TriggerListBox_States->getListItem(index);
						}
						else
						{
							//TODO: msg: "First select state"
						}
					}
					//add action button m_PickButton_AddAction is used in both pick and trigger dialogs - is it ok?
					else if(m_wnd_AddTriggerScriptAction && (m_PickButton_AddAction->getID() == event.GUIEvent.Caller->getID()))
					{
						s32 index = m_TriggerListBox_States->getSelected();
						if(index>-1)
						{
							TAction action;
							action.name = m_PickComboBox_Actions->getText();
							if(action.name != stringw(L"none"))
							{
								action.attribute = m_PickEditBox_ActionAttribute->getText();
								action.value = m_PickEditBox_ActionValue->getText();
								action.target = m_PickComboBox_ActionTarget->getText();
								if (action.target == stringw(L"id"))
								{
									if(stringw(m_PickEditBox_ActionTargetID->getText()) != stringw(L""))
									{
										action.target = m_PickEditBox_ActionTargetID->getText();
										m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.push_back(action);
										WriteTriggerScript(index);
									}
									else
									{
										//TODO: message: ENTER ID Value for target of this action
									}
								}
								else
								{
									m_EditorManager->getScriptEngine()->SelectedListOfScriptActions[index].actions.push_back(action);
									WriteTriggerScript(index);
								}
							}
							else
							{
								//TODO: message: "select action first!"
							}
						}
						else
						{
							//TODO message: select event first!
						}
					}
					else if (m_wnd_AddTriggerScriptAction && (m_PickButton_Save->getID() == event.GUIEvent.Caller->getID()))
					{
						SaveTriggerScript(m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->script.c_str());
						m_wnd_AddTriggerScriptAction->remove();
						m_wnd_AddTriggerScriptAction = NULL;
					}
					else if (m_wnd_AddTriggerScriptAction && (m_PickButton_Cancel->getID() == event.GUIEvent.Caller->getID()))
					{
						m_wnd_AddTriggerScriptAction->remove();
						m_wnd_AddTriggerScriptAction = NULL;
					}
					//Button clicked: AddNPCScriptAction MakeNPCScriptWindow
					else if (!m_wnd_AddNPCScriptAction && !m_wnd_containerContent && !m_wnd_AddPickScriptAction && 
						m_NPCScriptButton->getID() == event.GUIEvent.Caller->getID())
					{
						m_EditorManager->getScriptEngine()->SelectedListOfScriptActions.clear();
						if(m_EditorManager->getEdiLevel()->SingleObjectSelected())
						{
							m_SelectedObjectID = m_EditorManager->getEdiLevel()->m_SelectedGameObject->getID();
							if ((m_SelectedObjectID >= 0) && (m_SelectedObjectID < (s32)m_EditorManager->getEdiLevel()->m_ListOfGameObjects.size()))
							{
								//create conversation dialog window
								stringw path = m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->mesh;
								stringw rootName = getRootNameFromPath(path);
								m_DialogFilename = rootName.c_str();
								m_DialogFilename += ".dlg";
								m_wnd_AddNPCScriptAction = MakeNPCScriptWindow(m_EditorManager,env,m_EditorManager->getDevice(),m_DialogFilename);
							}
						}
					}
					//add dialog node
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_AddNode->getID() == event.GUIEvent.Caller->getID())
					{
						//show dialog
						//m_NumNodes = m_DialogTree->GetNumItems();
						m_SelectedNode = m_DialogTree->GetSelected();
						if(m_SelectedNode)
						{
							if(m_SelectedNode->Player)
							{
								//add what NPC says next
								m_wnd_AddNPCNewText = env->addWindow(core::rect<s32>(300,250,500,350), true, L"Add NPC reply", 0, 6800);
								env->addStaticText(L"Add what NPC says next:", core::rect<s32>(5,30,200,40), false, false, m_wnd_AddNPCNewText, 6801, false);
								m_DialogEditBox_Text = env->addEditBox(L"", core::rect<s32>(5,45,200,65), true, m_wnd_AddNPCNewText, 6802);
								m_DialogButton_AddText = env->addButton(core::rect<s32>(80,75,120,95), m_wnd_AddNPCNewText, 6803, L"Ok");
								env->setFocus(m_DialogEditBox_Text);
							}
							else
							{
								if (!m_SelectedNode->isLink)
								{
									//add what Player says next
									m_wnd_AddNPCNewText = env->addWindow(core::rect<s32>(300,250,500,350), true, L"Add Player reply", 0, 6800);
									env->addStaticText(L"Add what Player says next:", core::rect<s32>(5,30,200,40), false, false, m_wnd_AddNPCNewText, 6801, false);
									m_DialogEditBox_Text = env->addEditBox(L"", core::rect<s32>(5,45,200,65), true, m_wnd_AddNPCNewText, 6802);
									m_DialogButton_AddText = env->addButton(core::rect<s32>(80,75,120,95), m_wnd_AddNPCNewText, 6803, L"Ok");
									env->setFocus(m_DialogEditBox_Text);
								}
								else
								{
									//cant add children to link nodes!
									m_wnd_AddNPCNewText = env->addWindow(core::rect<s32>(300,250,500,350), true, L"Insert Node Error", 0, 6800);
									env->addStaticText(L"Can not add children LINK node!", core::rect<s32>(5,30,200,40), false, false, m_wnd_AddNPCNewText, 6801, false);
									m_DialogButton_Warning = env->addButton(core::rect<s32>(80,65,120,85), m_wnd_AddNPCNewText, 6804, L"Ok");
									env->setFocus(m_DialogButton_Warning);
								}
							}
						}
						else
						{
							//select node first!
							m_wnd_AddNPCNewText = env->addWindow(core::rect<s32>(300,250,500,350), true, L"Select Node", 0, 6800);
							env->addStaticText(L"Select Parent Node First!", core::rect<s32>(5,30,200,40), false, false, m_wnd_AddNPCNewText, 6801, false);
							m_DialogButton_Warning = env->addButton(core::rect<s32>(80,65,120,85), m_wnd_AddNPCNewText, 6804, L"Ok");
							env->setFocus(m_DialogButton_Warning);
						}
					}
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_RemoveNode->getID() == event.GUIEvent.Caller->getID())
					{
						//remove node
						m_SelectedNode = m_DialogTree->GetSelected();
						u32 Id = m_SelectedNode->ID;
						if(m_SelectedNode)
						{
							if(m_SelectedNode != m_DialogTree->GetRootNode()->Child)
							{
								m_DialogTree->DeleteNode(m_SelectedNode);
							}
							else
							{
								//can not delete root node!
							}
						}
						else
						{
							//nothing selected!
						}
						//This is a dangerous line, if we reduce m_NumNodes, then adding new node will have id of existing last node (if last was not deleted)
						//this is why m_NumNodes is actualy "number of latest, ever increasing, ID number"
						//m_NumNodes = m_DialogTree->GetNumItems();

						//this condition should always be true, but we check anyway because it would crash
						if(m_DialogTree->GetRootNode()->Child)
						{
							//check if there are actions pointing to deleted node
							CheckDialogNodeActions(m_DialogTree->GetRootNode()->Child,Id);
						}
					}
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_Ok->getID() == event.GUIEvent.Caller->getID())
					{
						//save and exit
						SaveDialog(m_DialogFilename);
						m_wnd_AddNPCScriptAction->remove();
						m_wnd_AddNPCScriptAction = 0;
					}
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_Cancel->getID() == event.GUIEvent.Caller->getID())
					{
						//exit
						m_wnd_AddNPCScriptAction->remove();
						m_wnd_AddNPCScriptAction = 0;
					}
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_LinkNodes->getID() == event.GUIEvent.Caller->getID())
					{
						m_SelectedNode = m_DialogTree->GetSelected();
						if(m_SelectedNode)
						{
							if(m_SelectedNode != m_DialogTree->GetRootNode()->Child)
							{
								if(m_SelectedNode->Link)
								{
									//can't add more then one link to one node
									m_wnd_AddNPCNewText = env->addWindow(core::rect<s32>(300,250,500,350), true, L"Link Node Error", 0, 6800);
									env->addStaticText(L"Can't add more then one link to one node!", core::rect<s32>(5,30,200,40), false, false, m_wnd_AddNPCNewText, 6801, false);
									m_DialogButton_Warning = env->addButton(core::rect<s32>(80,65,120,85), m_wnd_AddNPCNewText, 6804, L"Ok");
									env->setFocus(m_DialogButton_Warning);
									return true;
								}
								m_bLinkDialogNodes = true;
							}
							else
							{
								//can't link RootNode
							}
						}
						else
						{
							//select node first!
							m_wnd_AddNPCNewText = env->addWindow(core::rect<s32>(300,250,500,350), true, L"Select Node", 0, 6800);
							env->addStaticText(L"Select Parent Node First!", core::rect<s32>(5,30,200,40), false, false, m_wnd_AddNPCNewText, 6801, false);
							m_DialogButton_Warning = env->addButton(core::rect<s32>(80,65,120,85), m_wnd_AddNPCNewText, 6804, L"Ok");
							env->setFocus(m_DialogButton_Warning);
						}
					}
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_ToggleEnabled->getID() == event.GUIEvent.Caller->getID())
					{
						m_SelectedNode = m_DialogTree->GetSelected();
						if(m_SelectedNode)
						{
							PropagateEnabled(m_SelectedNode,!m_SelectedNode->isEnabled);
						}
						else
						{
							//Nothing selected!
						}
					}
					//change node text
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_ChangeText->getID() == event.GUIEvent.Caller->getID())
					{
						//get selected node
						m_SelectedNode = m_DialogTree->GetSelected();
						stringw newText = m_DialogEditBox_ChangeText->getText();
						if(m_SelectedNode)
						{
							if(m_SelectedNode != m_DialogTree->GetRootNode()->Child)
							{
								if(newText != stringw(L""))
								{
									m_SelectedNode->Text = m_DialogEditBox_ChangeText->getText();
								}
								else
								{
									//Enter new text first!
									//Text Edito field is empty
								}
							}
							else
							{
								//Can't change text of the RootNode
							}
						}
						else
						{
							//Nothing selected!
							//select node first
						}
					}
					//add npc dialog action
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_AddAction->getID() == event.GUIEvent.Caller->getID())
					{
						m_SelectedNode = m_DialogTree->GetSelected();
						if(m_SelectedNode)
						{
							if(m_SelectedNode != m_DialogTree->GetRootNode()->Child)
							{
								//add action to selected node
								TAction action;
								action.name = m_DialogComboBox_Actions->getText();
								if(action.name != stringw(L"none"))
								{
									action.attribute = m_DialogEditBox_ActionAttribute->getText();
									action.value = m_DialogEditBox_ActionValue->getText();
									action.target = m_DialogComboBox_ActionTarget->getText();
									if (action.target == stringw(L"id"))
									{
										if(stringw(m_DialogEditBox_ActionTargetID->getText()) != stringw(L""))
										{
											action.target = m_DialogEditBox_ActionTargetID->getText();
											m_SelectedNode->scriptAction.actions.push_back(action);
											m_SelectedNode->hasScriptAction = true;
											WriteNPCDialogScriptAction();
										}
										else
										{
											//TODO: message: ENTER ID Value for target of this action
										}
									}
									else
									{
										m_SelectedNode->scriptAction.actions.push_back(action);
										m_SelectedNode->hasScriptAction = true;
										WriteNPCDialogScriptAction();
									}
								}
								else
								{
									//TODO: message: "select action first!"
								}
							}
							else
							{
								//cant add actions to RootNode, its not visible as option in dialog
							}
						}
						else
						{
							//Nothing selected!
						}
					}
					//clear dialog node action
					else if (m_wnd_AddNPCScriptAction && m_DialogButton_ClearAction->getID() == event.GUIEvent.Caller->getID())
					{
						m_SelectedNode = m_DialogTree->GetSelected();
						if(m_SelectedNode)
						{
							m_SelectedNode->scriptAction.actions.clear();
							m_SelectedNode->hasScriptAction = false;
							WriteNPCDialogScriptAction();
						}
					}
					else if (m_wnd_AddNPCNewText && m_DialogButton_AddText && m_DialogButton_AddText->getID() == event.GUIEvent.Caller->getID())
					{
						//Add new text node to dialog
						TreeNode* node = m_DialogTree->InsertNode(m_DialogEditBox_Text->getText(),m_SelectedNode,m_NumNodes+1);
						if(node)
						{
							m_NumNodes++;
							if(m_SelectedNode != m_DialogTree->GetRootNode())
							{
								node->Player = !m_SelectedNode->Player;
							}
							else
							{
								node->Player = false;
							}
							if(!node->Player)
							{
								node->NPCName = m_EditorManager->getEdiLevel()->m_ListOfGameObjects[m_SelectedObjectID]->name;
							}
							m_wnd_AddNPCNewText->remove();
							m_DialogButton_Warning = 0;
							m_DialogButton_AddText = 0;
							m_wnd_AddNPCNewText = 0;
							//env->setFocus(m_DialogButton_AddNode);
						}
						else
						{
							//error adding node!
						}
					}
					else if (m_wnd_AddNPCNewText && m_DialogButton_Warning && m_DialogButton_Warning->getID() == event.GUIEvent.Caller->getID())
					{
						//remove warning dialog window
						m_wnd_AddNPCNewText->remove();
						m_wnd_AddNPCNewText = 0;
						m_DialogButton_Warning = 0;
						m_DialogButton_AddText = 0;
						env->setFocus(m_DialogButton_AddNode);
					}
					//Button clicked: AddMonsterScriptAction MakeMonsterScriptWindow
					else if (m_MonsterScriptButton->getID() == event.GUIEvent.Caller->getID())
					{
					}
				}
				break;
			}
		}
	
	return false;
}

void CEditorGUI::SetPosition(stringw position)
{
	m_PropEditBox_Pos->setText(position.c_str());
}

/**
 * \brief Sets properties of selected object in GUI window.
 * When user selects game object (3D model) this function is called to fill out properties fields in 
 * Properties floating window. Special structure TGameObjectProperty is used for this data transfer.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::SetProperties(TGameObjectProperty properties)
{
	m_PropEditBox_Name->setText(properties.name.c_str());
	m_PropEditBox_Pos->setText(properties.position.c_str());
	m_PropEditBox_Model->setText(properties.mesh.c_str());
	m_PropEditBox_ID->setText(stringw(properties.id).c_str());
	m_PropEditBox_Description->setText(properties.description.c_str());

	if (properties.isVisible)
		m_PropComboBox_Visible->setSelected(0); //combo true
	else
		m_PropComboBox_Visible->setSelected(1); //combo false

	if (properties.isContainer)
	{
		m_PropComboBox_Container->setSelected(0);
		m_ContainerButton->setEnabled(true);
	}
	else
	{
		m_PropComboBox_Container->setSelected(1);
		m_ContainerButton->setEnabled(false);
	}
	if (properties.isPickable)
	{
		m_PropComboBox_Pickable->setSelected(0);
		m_PickScriptButton->setEnabled(true);
	}
	else
	{
		m_PropComboBox_Pickable->setSelected(1);
		m_PickScriptButton->setEnabled(false);
	}
	if (properties.isTrigger)
	{
		m_PropComboBox_Trigger->setSelected(0);
		m_TriggerScriptButton->setEnabled(true);
	}
	else
	{
		m_PropComboBox_Trigger->setSelected(1);
		m_TriggerScriptButton->setEnabled(false);
	}
	if (properties.isNPC)
	{
		m_PropComboBox_NPC->setSelected(0);
		m_NPCScriptButton->setEnabled(true);
	}
	else
	{
		m_PropComboBox_NPC->setSelected(1);
		m_NPCScriptButton->setEnabled(false);
	}
	if (properties.isMonster)
	{
		m_PropComboBox_Monster->setSelected(0);
		m_MonsterScriptButton->setEnabled(true);
	}
	else
	{
		m_PropComboBox_Monster->setSelected(1);
		m_MonsterScriptButton->setEnabled(false);
	}
}

/**
 * \brief Sets all properties to empty (or default) values.
 * When user deselects all game object (when no object is selected) this function is called to clear properties fields in 
 * Properties floating window.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::ClearProperties()
{
	m_PropEditBox_Name->setText(L"");
	m_PropEditBox_Pos->setText(L"");
	m_PropEditBox_Model->setText(L"");
	m_PropEditBox_ID->setText(L"");
	m_PropEditBox_Description->setText(L"");

	m_PropComboBox_Visible->setSelected(0); //combo true
	m_PropComboBox_Container->setSelected(1); //combo false
	m_PropComboBox_Pickable->setSelected(1); //combo false
	m_PropComboBox_Trigger->setSelected(1); //combo false
	m_PropComboBox_NPC->setSelected(1); //combo false
	m_PropComboBox_Monster->setSelected(1); //combo false
	m_ContainerButton->setEnabled(false);
	m_PickScriptButton->setEnabled(false);
	m_TriggerScriptButton->setEnabled(false);
	m_NPCScriptButton->setEnabled(false);
	m_MonsterScriptButton->setEnabled(false);
}

stringw CEditorGUI::getRootNameFromPath(stringw path)
{
	stringw rootName = path.subString(path.findLast('/')+1,path.size());
	rootName = rootName.subString(0,rootName.findLast('.'));
	return rootName;
}

/**
 * \brief Finds Icon file and loads 2d image to container object.
 * This function is called when user wants to add content to container.
 * When he clicks on Item, icon of that Item is moved to the container.
 * This can be used later for draging icons around to other containers.
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::SetIconAtHand(stringw parent, stringw name)
{
	bool icon_found = false;
	stringc dir = "media/Icons/";
	//find file in the filesystem
	IFileSystem* fs = m_EditorManager->getDevice()->getFileSystem();
	IFileList* filelist = NULL;
	fs->changeWorkingDirectoryTo(dir.c_str());
	filelist = fs->createFileList();
	stringc temp_filename;
	//we go through list of all files in given Folder
	for ( u32 a = 2; a < filelist->getFileCount(); a++) //start from nuber 2 because 0 and 1 are directories "." and ".."
	{
		if(!filelist->isDirectory(a))
		{
			temp_filename = filelist->getFileName(a);
			//check every file if containst our icon name
			if (temp_filename.find(name.c_str()) != -1)
			{
				//but accept only image files (add more later...)
				if (temp_filename.find(".gif") + temp_filename.find(".jpg") + temp_filename.find(".bmp") + temp_filename.find(".png") > -4) 
				{
					icon_found = true;
					break;
				}
			}
		}
	}
	fs->changeWorkingDirectoryTo("../../");

	if(icon_found)
	{
		//Add Icon to container
		m_pPreviewPickableItem = new CPickableItem();
		dir += temp_filename.c_str();
		m_pPreviewPickableItem->setName(dir.c_str());
		m_pPreviewPickableItem->m_Script = getRootNameFromPath(dir.c_str());
		m_pPreviewPickableItem->m_Script += L".script";
		m_pPreviewPickableItem->m_IconTexture = m_EditorManager->getGUIEnvironment()->getVideoDriver()->getTexture(dir.c_str());
		m_SelectedPickItemIcon_Texture = m_pPreviewPickableItem->m_IconTexture;
		SendModelToPickSession(parent, name);
		m_pContainer_EditPickItemScript->setEnabled(true);

		if(m_bAddPickDirectly)
		{
			IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(1357);
			container->InsertItem(m_pPreviewPickableItem);
		}
		else
		{
			m_pContainer_AddPickItemToContainer->setEnabled(true);
		}
	}
	else
	{
		//shibMessage("Unfortunately, this icon is not found");
		stringw message = "File ";
		message += name.c_str();
		message += " is missing from folder media/GameObjects/Pickables/Icons";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"File is missing!", message.c_str());
	}

	filelist->drop();
}

/**
 * \brief Loads 3D model and its properties and place it under the mouse pointer.
 * This lovely function is called when user clicks on element in the tree of game objects.
 * It makes sure GO exists in the file system, reads GO's parameters from XML, 
 * and opens GO's 3D model in EdLevel view. And sets GO's parameters in parameter window.
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::SendModelToPickSession(stringw parent, stringw name)
{
	bool model_found = false;
	stringc dir;
	IGUITabControl* tabGameObjects = (IGUITabControl*) m_wnd_gameItems->getElementFromId(6000,true);
	int a = tabGameObjects->getActiveTab();
	IGUITab* tabs = tabGameObjects->getTab(a);
	stringw Folder = "media/GameObjects/Pickables/";
	if (parent != "Root Node")
	{
		Folder += parent;
		Folder += "/";
	}
	//find file in the filesystem
	IFileList* filelist = NULL;
	dir = Folder.c_str();
	m_EditorManager->getFS()->changeWorkingDirectoryTo(dir.c_str());
	filelist = m_EditorManager->getFS()->createFileList();
	stringw temp_filename;
	//we go through list of all files in given Folder
	for ( u32 a = 2; a < filelist->getFileCount(); a++)
	{
		if(!filelist->isDirectory(a))
		{
			temp_filename = filelist->getFileName(a);
			//check every file if containst our model name
			if (temp_filename.find(name.c_str()) != -1)
			{
				//but exclude image files and other non model files (this is risky cause there can be a model file containing other models name, but it will be fixed later...
				if (temp_filename.find(".gif") + temp_filename.find(".jpg") + temp_filename.find(".bmp") + temp_filename.find(".png") + temp_filename.find(".xml") < -4) 
				{
					model_found = true;
					break;
				}
			}
		}
	}
	m_EditorManager->backToWorkingDirectory();
	if(model_found)
	{
		//Load properties from xml file if existing
		stringc jo = dir.c_str();
		jo += name.c_str();
		TGameObjectProperty propertiz;
		propertiz.name = name;
		propertiz.mesh = Folder + temp_filename;
		propertiz.isContainer = false;
		propertiz.isMonster = false;
		propertiz.isNPC = false;
		propertiz.isPickable = false;
		propertiz.isTrigger = false;
		propertiz.isVisible = true;
		propertiz.description = L"No description specified";
		propertiz.script = L"";
		jo += ".xml"; //assume there is xml file with the same name as model
		io::IXMLReader* xml = m_EditorManager->getFS()->createXMLReader(jo.c_str());
		if (xml)
		{
			while(xml->read())
			{
				switch(xml->getNodeType())
				{
				case io::EXN_ELEMENT:
					{
						stringw figo;
						if (stringw("Name") == xml->getNodeName())
						{
							propertiz.name = xml->getAttributeValue(L"value");
						}
						else if (stringw("Pickable") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isPickable = true;
							}
							else
							{
								propertiz.isPickable = false;
							}
						}
						else if (stringw("Container") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isContainer = true;
							}
							else
							{
								propertiz.isContainer = false;
							}
						}
						else if (stringw("Monster") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isMonster = true;
							}
							else
							{
								propertiz.isMonster = false;
							}
						}
						else if (stringw("Trigger") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isTrigger = true;
							}
							else
							{
								propertiz.isTrigger = false;
							}
						}
						else if (stringw("Description") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							propertiz.description = figo.c_str();
						}
						else if (stringw("Script") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							propertiz.script = figo.c_str();
						}
						else if (stringw("Visible") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isVisible = true;
							}
							else
							{
								propertiz.isVisible = false;
							}
						}
					}
				}
			}
		}

		//Tell editor wich model to load 
		stringc filename = propertiz.mesh.c_str();
		bool success = m_EditorManager->getEdiLevel()->SendModelToPickSession(filename,propertiz);

		//finally, set collected properties
		m_pPickPreviewDescriptionText->setText(propertiz.description.c_str());
		//SetPreviewProperties(propertiz);

		if(xml)
			xml->drop();
	}
	else
	{
		//shibMessage("Unfortunately, this model is not found");
		stringw message = "File ";
		message += name.c_str();
		message += " is missing from folder ";
		message += Folder.c_str();
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"File is missing!", message.c_str());
	}

	m_EditorManager->backToWorkingDirectory();
	filelist->drop();
}

/**
 * \brief Loads 3D model and its properties and place it under the mouse pointer.
 * This lovely function is called when user clicks on element in the tree of game objects.
 * It makes sure GO exists in the file system, reads GO's parameters from XML, 
 * and opens GO's 3D model in EdLevel view. And sets GO's parameters in parameter window.
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::SetElementAtHand(stringw parent, stringw name)
{
	bool model_found = false;
	m_bElementPicking = false;
	stringc dir;
	IGUITabControl* tabGameObjects = (IGUITabControl*) m_wnd_gameItems->getElementFromId(6000,true);
	int a = tabGameObjects->getActiveTab();
	IGUITab* tabs = tabGameObjects->getTab(a);
	stringw Folder = "media/GameObjects/";
	Folder += tabs->getText();
	Folder += "/";
	if (parent != "Root Node")
	{
		Folder += parent;
		Folder += "/";
	}
	//find file in the filesystem
	IFileList* filelist = NULL;
	dir = Folder.c_str();
	m_EditorManager->getFS()->changeWorkingDirectoryTo(dir.c_str());
	filelist = m_EditorManager->getFS()->createFileList();
	stringw temp_filename;
	//we go through list of all files in given Folder
	for ( u32 a = 2; a < filelist->getFileCount(); a++)
	{
		if(!filelist->isDirectory(a))
		{
			temp_filename = filelist->getFileName(a);
			//check every file if containst our model name
			if (temp_filename.find(name.c_str()) != -1)
			{
				//but exclude image files and other non model files (this is risky cause there can be a model file containing other models name, but it will be fixed later...
				if (temp_filename.find(".jpg") + temp_filename.find(".bmp") + temp_filename.find(".png") + temp_filename.find(".xml") < -3) 
				{
					model_found = true;
					break;
				}
			}
		}
	}
	m_EditorManager->backToWorkingDirectory();
	if(model_found)
	{
		//Load properties from xml file if existing
		stringc jo = dir.c_str();
		jo += name.c_str();
		TGameObjectProperty propertiz;
		propertiz.name = name;
		propertiz.mesh = Folder + temp_filename;
		propertiz.isContainer = false;
		propertiz.isMonster = false;
		propertiz.isNPC = false;
		propertiz.isPickable = false;
		propertiz.isTrigger = false;
		propertiz.isVisible = true;
		propertiz.description = L"No description specified";
		propertiz.script = getRootNameFromPath(propertiz.mesh);
		propertiz.script += L".script";
		jo += ".xml"; //assume there is xml file with the same name as model
		io::IXMLReader* xml = m_EditorManager->getFS()->createXMLReader(jo.c_str());
		if (xml)
		{
			while(xml->read())
			{
				switch(xml->getNodeType())
				{
				case io::EXN_ELEMENT:
					{
						stringw figo;
						if (stringw("Name") == xml->getNodeName())
						{
							propertiz.name = xml->getAttributeValue(L"value");
						}
						else if (stringw("Pickable") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isPickable = true;
							}
							else
							{
								propertiz.isPickable = false;
							}
						}
						else if (stringw("Container") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isContainer = true;
							}
							else
							{
								propertiz.isContainer = false;
							}
						}
						else if (stringw("Monster") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isMonster = true;
							}
							else
							{
								propertiz.isMonster = false;
							}
						}
						else if (stringw("NPC") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isNPC = true;
							}
							else
							{
								propertiz.isNPC = false;
							}
						}
						else if (stringw("Trigger") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isTrigger = true;
							}
							else
							{
								propertiz.isTrigger = false;
							}
						}
						else if (stringw("Description") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							propertiz.description = figo.c_str();
						}
						else if (stringw("Script") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							propertiz.script = figo.c_str();
						}
						else if (stringw("Visible") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isVisible = true;
							}
							else
							{
								propertiz.isVisible = false;
							}
						}
					}
				}
			}
		}

		//Tell editor wich model to load 
		stringc filename = propertiz.mesh.c_str();
		bool success = m_EditorManager->getEdiLevel()->SetElementAtHand(filename,propertiz);
		if (success)
		{
			propertiz.id = m_EditorManager->m_ID;
			m_EditorManager->m_ID++;
		}

		//finally, set collected properties
		SetProperties(propertiz);

		if(xml)
			xml->drop();

	}
	else
	{
		//shibMessage("Unfortunately, this model is not found");
		stringw message = "File ";
		message += name.c_str();
		message += " is missing from folder ";
		message += Folder.c_str();
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"File is missing!", message.c_str());

	}

	m_EditorManager->backToWorkingDirectory();
	filelist->drop();
}

/**
 * \brief Loads 3D model and its properties and place it under the mouse pointer.
 * This lovely function is called when user clicks on element in the tree of game objects.
 * It makes sure GO exists in the file system, reads GO's parameters from XML, 
 * and opens GO's 3D model in EdLevel view. And sets GO's parameters in parameter window.
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorGUI::SendModelToPhotoSession(stringw parent, stringw name)
{
	bool model_found = false;
	stringc dir;
	IGUITabControl* tabGameObjects = (IGUITabControl*) m_wnd_gameItems->getElementFromId(6000,true);
	int a = tabGameObjects->getActiveTab();
	IGUITab* tabs = tabGameObjects->getTab(a);
	stringw Folder = "media/GameObjects/";
	Folder += tabs->getText();
	Folder += "/";
	if (parent != "Root Node")
	{
		Folder += parent;
		Folder += "/";
	}
	//find file in the filesystem
	IFileList* filelist = NULL;
	dir = Folder.c_str();
	m_EditorManager->getFS()->changeWorkingDirectoryTo(dir.c_str());
	filelist = m_EditorManager->getFS()->createFileList();
	stringw temp_filename;
	//we go through list of all files in given Folder
	for ( u32 a = 2; a < filelist->getFileCount(); a++)
	{
		if(!filelist->isDirectory(a))
		{
			temp_filename = filelist->getFileName(a);
			//check every file if containst our model name
			if (temp_filename.find(name.c_str()) != -1)
			{
				//but exclude image files and other non model files (this is risky cause there can be a model file containing other models name, but it will be fixed later...
				if (temp_filename.find(".jpg") + temp_filename.find(".bmp") + temp_filename.find(".png") + temp_filename.find(".xml") < -3) 
				{
					model_found = true;
					break;
				}
			}
		}
	}
	m_EditorManager->backToWorkingDirectory();
	if(model_found)
	{
		//Load properties from xml file if existing
		stringc jo = dir.c_str();
		jo += name.c_str();
		TGameObjectProperty propertiz;
		propertiz.name = name;
		propertiz.mesh = Folder + temp_filename;
		propertiz.isContainer = false;
		propertiz.isMonster = false;
		propertiz.isNPC = false;
		propertiz.isPickable = false;
		propertiz.isTrigger = false;
		propertiz.isVisible = true;
		propertiz.description = L"No description specified";
		propertiz.script = getRootNameFromPath(propertiz.mesh);
		propertiz.script += L".script";
		jo += ".xml"; //assume there is xml file with the same name as model
		io::IXMLReader* xml = m_EditorManager->getFS()->createXMLReader(jo.c_str());
		if (xml)
		{
			while(xml->read())
			{
				switch(xml->getNodeType())
				{
				case io::EXN_ELEMENT:
					{
						stringw figo;
						if (stringw("Name") == xml->getNodeName())
						{
							propertiz.name = xml->getAttributeValue(L"value");
						}
						else if (stringw("Pickable") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isPickable = true;
							}
							else
							{
								propertiz.isPickable = false;
							}
						}
						else if (stringw("Container") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isContainer = true;
							}
							else
							{
								propertiz.isContainer = false;
							}
						}
						else if (stringw("Monster") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isMonster = true;
							}
							else
							{
								propertiz.isMonster = false;
							}
						}
						else if (stringw("Trigger") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isTrigger = true;
							}
							else
							{
								propertiz.isTrigger = false;
							}
						}
						else if (stringw("Description") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							propertiz.description = figo.c_str();
						}
						else if (stringw("Script") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							propertiz.script = figo.c_str();
						}
						else if (stringw("Visible") == xml->getNodeName())
						{
							figo = xml->getAttributeValue(L"value");
							if (figo == L"true")
							{
								propertiz.isVisible = true;
							}
							else
							{
								propertiz.isVisible = false;
							}
						}
					}
				}
			}
		}

		//Tell editor wich model to load 
		stringc filename = propertiz.mesh.c_str();
		bool success = m_EditorManager->getEdiLevel()->SendModelToPhotoSession(filename,propertiz);

		//finally, set collected properties
		m_pPreviewDescriptionText->setText(propertiz.description.c_str());
		//SetPreviewProperties(propertiz);
	}
	else
	{
		//shibMessage("Unfortunately, this model is not found");
		stringw message = "File ";
		message += name.c_str();
		message += " is missing from folder ";
		message += Folder.c_str();
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"File is missing!", message.c_str());
	}

	m_EditorManager->backToWorkingDirectory();
	filelist->drop();
}

/**
 * \brief Initializes all windows and gui elements...
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorGUI::Init(CEditorManager* edMngr)
{
	m_EditorManager = edMngr;

	//Setting GUI transparency (alpha)
	for (s32 i=0; i<irr::gui::EGDC_COUNT ; ++i)
	{
		video::SColor col = m_EditorManager->getGUIEnvironment()->getSkin()->getColor((EGUI_DEFAULT_COLOR)i);
		col.setAlpha(200);
		m_EditorManager->getGUIEnvironment()->getSkin()->setColor((EGUI_DEFAULT_COLOR)i, col);
	}

	/*Create Menu*/
	gui::IGUIContextMenu* menu = m_EditorManager->getGUIEnvironment()->addMenu();
	//menu->
	menu->addItem(L"File", -1, true, true);
	menu->addItem(L"Edit", -1, true, true);
	menu->addItem(L"View", -1, true, true);
	menu->addItem(L"Help", -1, true, true);

	gui::IGUIContextMenu* submenu;
	submenu = menu->getSubMenu(0);
	submenu->addItem(L"New Level", 100);
	submenu->addItem(L"Open Level...", 101);
	submenu->addItem(L"Save Level", 102);
	submenu->addItem(L"Save Level As...", 103);
	submenu->addSeparator();
	submenu->addItem(L"Quit", 200);

	submenu = menu->getSubMenu(1);
	submenu->addItem(L"Undo", 300);
	submenu->addItem(L"Redo", -1, true, false);
	submenu->addSeparator();
	submenu->addItem(L"Select All", -1, true, false );

	submenu = menu->getSubMenu(2);
	submenu->addItem(L"Object Properties", 400, true, false, true);
	submenu->addItem(L"Game Objects", 410, true, false, true);

	submenu = menu->getSubMenu(3);
	submenu->addItem(L"About", 500);

	/*Properties window*/
	m_wnd_properties = MakePropertiesWindow(m_EditorManager, m_EditorManager->getGUIEnvironment(), m_EditorManager->getDevice());

	/*Game Objects window*/
	m_wnd_gameItems = MakeGameObjectsWindow(m_EditorManager->getGUIEnvironment(), m_EditorManager->getDevice());

	//Preview models with render on texture
	if (m_EditorManager->getDriver()->queryFeature(video::EVDF_RENDER_TO_TARGET))
	{
		// add fixed cameras
		m_PhotoCamera = m_EditorManager->getSceneMngr()->addCameraSceneNode(0, 
			core::vector3df(0,-10000,0),
			core::vector3df(0,-10000,-100));
		m_PickCamera = m_EditorManager->getSceneMngr()->addCameraSceneNode(0, 
			core::vector3df(0,-10000,0),
			core::vector3df(0,-10000,100));

		// add textures to render on
		m_TextureToRenderOn = m_EditorManager->getDriver()->createRenderTargetTexture(core::dimension2di(200,200));
		m_ContainerPickPreview_TextureToRenderOn = m_EditorManager->getDriver()->createRenderTargetTexture(core::dimension2di(180,180));
		
		//model preview window
		m_wnd_ModelPreview = m_EditorManager->getGUIEnvironment()->addWindow(core::rect<s32>(0,410,230,750), false, L"Object Preview", 0, 4800);
		m_EditorManager->getGUIEnvironment()->addStaticText(L"Description:", core::rect<s32>(10,245,210,255), false, false, m_wnd_ModelPreview, 4560, false);
		m_pPreviewDescriptionText = m_EditorManager->getGUIEnvironment()->addStaticText(L"Description:", core::rect<s32>(10,255,210,315), true, true, m_wnd_ModelPreview, 4561, false);
	}
	printf("GUI Initialized");

	return true;
}