/** 
 * \file EdLevel.cpp
 * \brief File EdLevel.cpp takes care of level structure - it hold together all game objects on the map.
 *
 * takes care of saving/loading levels 
 * manages game object arrays (adding deleting game objects)
 * allows translation, rotation and scaling of models.
 * implements undo/redo functionality
 * translates events to game objects -> on mouse click, on rotate etc..
 * tightly communicates with GUI environment, for example: 
 *  - selecting different game object causes changes in GUI window Properties
 *  - changed game object properties must be saved to map file
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "EDLevel.h"
#include "../EditorManager.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CEditorLevel::CEditorLevel()
{
	m_EditorManager = NULL;
	m_SelectedGameObject = NULL;
	m_PreviewGameObject = 0;
	m_PreviewPickGameObject = 0;
	m_bMoveSelectedNode = false;
	m_bRotateSelectedNode = false;
	m_bShiftPressed = false;
	m_bElementAtHand = false;
	m_CurrentZoom = vector3df(1.0f,1.0f,1.0f);
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CEditorLevel::~CEditorLevel()
{
}

/**
 * \brief Used to set Map name without directory path and extension.
 * Map name is used to create name of container content xml file.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorLevel::SetMapName(stringc mapname)
{
	int position1 = mapname.findLast('\\');
	int position2 = mapname.findLast('/');
	if( position1 < position2)
		position1 = position2;
	m_MapName = mapname.subString(position1+1,mapname.size()-position1 - 5);
}

void CEditorLevel::SetLoadingDir(stringc mapname)
{
	m_LoadingDir = mapname;
}

/*
 * createUserData is irrlicht serialization function used for adding user defined attributes to 
 * irr scenes (xml files) for each scene node. In this case, information about nodes like
 * is it a container object or does it responds with dialogue are stored in this function.
 * This function has 'reverse' version for loading user data from irr scene called: OnReadUserData
 */
IAttributes* CEditorLevel::createUserData (irr::scene::ISceneNode *sceneNode)
{
	IAttributes* attr = m_EditorManager->getFS()->createEmptyAttributes(m_EditorManager->getDriver());

	//for each node, add this lovely, useless attribute:
	attr->addString("Homepage",L"www.mystic-peanut.com"); //(valuable testing purpose)

	if(sceneNode->getType() != ESNT_TERRAIN)
	{
		//For each scene node, not terrain or camera (although this will be changed)
		if((sceneNode->getID() >= 0) && (sceneNode->getID() < (s32)m_ListOfGameObjects.size()))
		{
			//If node is container, add user data attribute, and create xml file for container content
			if (m_ListOfGameObjects[sceneNode->getID()]->isContainer)
			{
				//adding attribute
				attr->addBool("isContainer",true);

				//Make xml file wich holds container content
				s32 id = sceneNode->getID();;
				stringw id_text = id;
				stringc container_filename = "CC_";
				container_filename += m_MapName.c_str();
				container_filename += "_";
				container_filename += id;
				container_filename += ".xml";
				io::IXMLWriter* xml = m_EditorManager->getDevice()->getFileSystem()->createXMLWriter(container_filename.c_str());

				if (xml)
				{
					xml->writeXMLHeader(); xml->writeLineBreak();
					xml->writeElement(L"Container",false,L"id",id_text.c_str()); xml->writeLineBreak();
					if ((id >= 0)&&(id<(s32)m_ListOfGameObjects.size()))
					for ( s32 index = 0; index < m_ListOfGameObjects[id]->GetNumberOfPickableItems(); index++)
					{
						stringw temp = m_ListOfGameObjects[id]->GetPickableItem(index).c_str();
						xml->writeElement(L"Pickables",true,L"iconname",temp.c_str());
						xml->writeLineBreak();
					}
					xml->writeClosingTag(L"Container"); xml->writeLineBreak();
					xml->drop(); // don't forget to delete the xml reader
				}
			}
			if (m_ListOfGameObjects[sceneNode->getID()]->isPickable)
			{
				//adding attribute isPickable
				attr->addBool("isPickable",true);
			}
			if (m_ListOfGameObjects[sceneNode->getID()]->isTrigger)
			{
				//adding atribute isTrigger
				attr->addBool("isTrigger",true);
			}
			if (m_ListOfGameObjects[sceneNode->getID()]->isNPC)
			{
				//adding atribute isNPC
				attr->addBool("isNPC",true);
			}
			if (m_ListOfGameObjects[sceneNode->getID()]->script != stringw(L""))
			{
				attr->addString("Script",m_ListOfGameObjects[sceneNode->getID()]->script.c_str());
			}
			if (m_ListOfGameObjects[sceneNode->getID()]->state != stringw(L""))
			{
				attr->addString("State",m_ListOfGameObjects[sceneNode->getID()]->state.c_str());
			}
		}
	}
	return attr; //returned structure will be saved to irr file by irrlicht automatically
}

/*
 * OnReadUserData is irrlicht serialization function used for loading user defined attributes 
 * from irr scenes (xml files) for each scene node. In this case, information about nodes like
 * is it a container object or does it responds with dialogue are loaded in this function.
 * This function has 'reverse' version for saving user data to irr scene called: createUserData
 */
void CEditorLevel::OnReadUserData (irr::scene::ISceneNode *sceneNode, irr::io::IAttributes *userData)
{
	m_EditorManager->getFS()->changeWorkingDirectoryTo(m_LoadingDir.c_str());
	//ignoring camera, terrain and other special nodes with ID = -1
	if(sceneNode->getID() >= 0)
	{
		//adding new game object to the map
		CGameObject* gameObject = new CGameObject();
		if(userData->getAttributeAsBool("isContainer"))
		{
			gameObject->isContainer = true;

			//Load Existing Container Content (if any)
			stringc container_filename = "CC_";
			container_filename += m_MapName.c_str();
			container_filename += "_";
			container_filename += sceneNode->getID();
			container_filename += ".xml";
			io::IXMLReader* xml = m_EditorManager->getDevice()->getFileSystem()->createXMLReader(container_filename.c_str());
			while(xml && xml->read())
			{
				//Load container content from xml file to GameObjects list of pickable items
				if (core::stringw("Pickables") == xml->getNodeName())
				{
					//Add this element
					stringc item = xml->getAttributeValue(L"iconname");
					gameObject->AddPickableItem(item);
				}
			}

			if (xml)
				xml->drop(); // don't forget to delete the xml reader
		}
		if(userData->getAttributeAsBool("isPickable"))
		{
			gameObject->isPickable = true;
		}
		if(userData->getAttributeAsBool("isTrigger"))
		{
			gameObject->isTrigger = true;
		}
		if(userData->getAttributeAsBool("isNPC"))
		{
			gameObject->isNPC = true;
		}
		gameObject->script = userData->getAttributeAsStringW("Script");
		gameObject->state = userData->getAttributeAsStringW("State");

		gameObject->id = m_EditorManager->m_ID;
		m_ListOfGameObjects.push_back(gameObject);
		if (m_EditorManager->m_ID != sceneNode->getID())
		{
			sceneNode->setID(m_EditorManager->m_ID); //This should never happen
			stringw message = "Node IDs are messed up. Map might not work properly.";
			m_EditorManager->getGUIEnvironment()->addMessageBox(L"Error loading map!", message.c_str());
		}

		m_EditorManager->m_ID++;
	}
	m_EditorManager->backToWorkingDirectory();
}

/**
 * helper function: draw the object aligned bounding box
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void drawObjectBoundingBox(scene::ISceneNode* node,
                                      video::IVideoDriver* driver,
                                      video::SColor color = video::SColor(255, 0, 255, 0))
{
   if (!node || !driver)
      return;

   video::SMaterial matl;
   matl.Lighting = false;
   driver->setMaterial(matl);
   driver->setTransform(video::ETS_WORLD, node->getAbsoluteTransformation());
   driver->draw3DBox(node->getBoundingBox(), color);

   	/*Vitek said:
	Be careful with getTransformedBoundingBox(). It doesn't rotate the box properly so you will occasionally get boxes that don't actually enclose the object. This can be very noticeable when you have an oblong object that is rotated.
	The best is to use matrix4::transformBoxEx() to do something like this...

	core::aabbox3df b1 = cube->getBoundingBox();
	cube->getAbsoluteTransformation().transformBoxEx(b1);
	core::aabbox3df b2 = sphere->getBoundingBox();
	sphere->getAbsoluteTransformation().transformBoxEx(b2);
	*/
}

/**
 * \brief So far used just for drawing bounding box.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorLevel::OnRender()
{
	if(m_PreviewGameObject && m_EditorManager)
	{
		m_PreviewGameObject->setRotation(core::vector3df(0,(f32)m_EditorManager->getDevice()->getTimer()->getTime()/10,0));
	}
	if(m_PreviewPickGameObject && m_EditorManager)
	{
		m_PreviewPickGameObject->setRotation(core::vector3df(0,(f32)m_EditorManager->getDevice()->getTimer()->getTime()/15,0));
	}
	if (m_SelectedGameObject && m_EditorManager)
	{
		drawObjectBoundingBox(m_SelectedGameObject,m_EditorManager->getDriver());
	}
}

/**
 * \brief Aahhh... SetElementAtHand. Creates 3D model from given filename and adds it to the scene.
 * \param filename name of the 3d model file
 * \param properties game object properties that will be written to irr scene node.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorLevel::SetElementAtHand(stringc filename, TGameObjectProperty properties)
{
	//If there was element under hand and another one was picked from the tree
	if(m_bElementAtHand && m_SelectedGameObject)
	{
		m_ListOfGameObjects.erase(m_SelectedGameObject->getID(),1);
		//delete the bastard
		m_SelectedGameObject->remove();
		m_SelectedGameObject = NULL;
		m_EditorManager->m_ID--;
	}

	IAnimatedMesh* m = m_EditorManager->getSceneMngr()->getMesh(filename.c_str());
	if (m)
	{
		m_bElementAtHand = true;
		CGameObject* gameObject = new CGameObject();
		m_SelectedGameObject = m_EditorManager->getSceneMngr()->addAnimatedMeshSceneNode(m);
		m_SelectedGameObject->setMaterialFlag(EMF_LIGHTING, false);
		//GameObject->setAnimationSpeed(100);
		
		//position object under mouse pointer (where mouse ray intersects terrain)
		m_bMoveSelectedNode = true;
		triangle3df instersection_triangle;
		ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_EditorManager->getSceneMngr()->getSceneNodeFromType(ESNT_TERRAIN);
		ITriangleSelector* selector = terrain->getTriangleSelector();
		line3d<f32> picking_line = m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getRayFromScreenCoordinates(m_EditorManager->getDevice()->getCursorControl()->getPosition());
		m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getCollisionPoint(picking_line,selector,m_instersection_point,instersection_triangle);
		m_selectedGameObjectStartMovingPosition = m_instersection_point;
		vector3df temp;
		temp.Y = m_instersection_point.Y;
		temp.X = 0;
		temp.Z = 0;
		m_SelectedGameObject->setPosition(temp);

		//save object properties
		stringc temp_param = properties.name.c_str();
		m_SelectedGameObject->setName(temp_param.c_str());
		m_SelectedGameObject->setID(m_EditorManager->m_ID);
		m_SelectedGameObject->setVisible(properties.isVisible);
		gameObject->mesh = properties.mesh;
		gameObject->name = temp_param.c_str();
		gameObject->isContainer = properties.isContainer;
		gameObject->isPickable = properties.isPickable;
		gameObject->isTrigger = properties.isTrigger;
		gameObject->isNPC = properties.isNPC;
		gameObject->isMonster = properties.isMonster;
		gameObject->id = m_EditorManager->m_ID;
		gameObject->description = properties.description.c_str();
		gameObject->script = properties.script.c_str();
		m_ListOfGameObjects.push_back(gameObject);
	}
	else
	{
		//allert message: "Model could not be loaded"
		stringw message = "Model ";
		message += filename.c_str();
		message += " can not be loaded.";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Error loading model!", message.c_str());
		return false;
	}

	return true;
}

void CEditorLevel::RemovePhotoSessionModel()
{
	if(m_PreviewGameObject)
	{
		m_PreviewGameObject->remove();
		m_PreviewGameObject = 0;
	}
}

/**
 * \brief Creates 3D model from given filename, adds it to the scene, and sends it to photo session.
 * \param filename name of the 3d model file
 * \param properties game object properties that will be written to irr scene node.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorLevel::SendModelToPhotoSession(stringc filename, TGameObjectProperty properties)
{
	IAnimatedMesh* m = m_EditorManager->getSceneMngr()->getMesh(filename.c_str());
	if (m)
	{
		if(m_PreviewGameObject)
			m_PreviewGameObject->remove();
		m_PreviewGameObject = m_EditorManager->getSceneMngr()->addAnimatedMeshSceneNode(m);
		if(m_PreviewGameObject)
		{
			m_PreviewGameObject->setMaterialFlag(EMF_LIGHTING, false);
			vector3df radius = m_PreviewGameObject->getBoundingBox().MaxEdge - m_PreviewGameObject->getBoundingBox().getCenter();
			radius *= 2;
			if ((radius.X < 20) && (radius.Y < 20) && (radius.Z < 20))
			{
				m_PreviewGameObject->setPosition(vector3df(0, -10010, -20));
			}
			else if ((radius.X < 50) && (radius.Y < 50) && (radius.Z < 50))
			{
				m_PreviewGameObject->setPosition(vector3df(0, -10020, -40));
			}
			else if ((radius.X < 110) && (radius.Y < 110) && (radius.Z < 110))
			{
				m_PreviewGameObject->setPosition(vector3df(0, -10040, -90));
			}
			else if ((radius.X < 190) && (radius.Y < 190) && (radius.Z < 190))
			{
				m_PreviewGameObject->setPosition(vector3df(0, -10070, -190));
			}
			else
			{
				m_PreviewGameObject->setPosition(vector3df(0, -10090, -390));
			}
		}
	}
	else
	{
		//allert message: "Model could not be loaded"
		stringw message = "Model ";
		message += filename.c_str();
		message += " can not be loaded.";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Error loading model!", message.c_str());
		return false;
	}

	return true;
}

void CEditorLevel::RemovePickSessionModel()
{
	if(m_PreviewPickGameObject)
	{
		m_PreviewPickGameObject->remove();
		m_PreviewPickGameObject = 0;
	}
}
/**
 * \brief Creates 3D model from given filename, adds it to the scene, and sends it to photo session.
 * \param filename name of the 3d model file
 * \param properties game object properties that will be written to irr scene node.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorLevel::SendModelToPickSession(stringc filename, TGameObjectProperty properties)
{
	IAnimatedMesh* m = m_EditorManager->getSceneMngr()->getMesh(filename.c_str());
	if (m)
	{
		if(m_PreviewPickGameObject)
			m_PreviewPickGameObject->remove();
		m_PreviewPickGameObject = m_EditorManager->getSceneMngr()->addAnimatedMeshSceneNode(m);
		if(m_PreviewPickGameObject)
		{
			m_PreviewPickGameObject->setMaterialFlag(EMF_LIGHTING, false);
			vector3df radius = m_PreviewPickGameObject->getBoundingBox().MaxEdge - m_PreviewPickGameObject->getBoundingBox().getCenter();
			radius *= 2;
			if ((radius.X < 20) && (radius.Y < 20) && (radius.Z < 20))
			{
				m_PreviewPickGameObject->setPosition(vector3df(0, -10010, 20));
			}
			else if ((radius.X < 50) && (radius.Y < 50) && (radius.Z < 50))
			{
				m_PreviewPickGameObject->setPosition(vector3df(0, -10020, 40));
			}
			else if ((radius.X < 110) && (radius.Y < 110) && (radius.Z < 110))
			{
				m_PreviewPickGameObject->setPosition(vector3df(0, -10040, 90));
			}
			else if ((radius.X < 190) && (radius.Y < 190) && (radius.Z < 190))
			{
				m_PreviewPickGameObject->setPosition(vector3df(0, -10070, 190));
			}
			else
			{
				m_PreviewPickGameObject->setPosition(vector3df(0, -10090, -390));
			}
		}
	}
	else
	{
		//allert message: "Model could not be loaded"
		stringw message = "Model ";
		message += filename.c_str();
		message += " can not be loaded.";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Error loading model!", message.c_str());
		return false;
	}

	return true;
}

/**
 * \brief Creates camera in initial position (used for loading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorLevel::CreateCamera(vector3df position)
{
	m_LevelCamera = new RTSCamera(m_EditorManager->getDevice(),m_EditorManager->getSceneMngr()->getRootSceneNode(),m_EditorManager->getSceneMngr(),-1,100.0f,10.0f,100.0f);
	m_LevelCamera->setPosition(position);
	//matrix4 projection;
	//projection.buildProjectionMatrixOrthoLH(300, 300, 3, 30000);
	//camera->setProjectionMatrix(projection);
	//camera->setIsOrthogonal(true);
}

/**
 * \brief Init function stores pointer to CEditorManager and creates camera and default terrain.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorLevel::Init(CEditorManager* edMngr)
{
	m_EditorManager = edMngr;

	m_LevelCamera = new RTSCamera(m_EditorManager->getDevice(),m_EditorManager->getSceneMngr()->getRootSceneNode(),m_EditorManager->getSceneMngr(),-1,100.0f,10.0f,100.0f);
	m_LevelCamera->setPosition(vector3df(100,300,100));
	//matrix4 projection;
	//projection.buildProjectionMatrixOrthoLH(300, 300, 3, 30000);
	//camera->setProjectionMatrix(projection);
	//camera->setIsOrthogonal(true);

	m_EditorManager->getDriver()->setTextureCreationFlag(video::ETCF_ALWAYS_32_BIT, true);

	/*
	Here comes the terrain renderer scene node: We add it just like any 
	other scene node to the scene using ISceneManager::addTerrainSceneNode(). 
	The only parameter we use is a file name to the heightmap we use. A heightmap
	is simply a gray scale texture. The terrain renderer loads it and creates 
	the 3D terrain from it.
	To make the terrain look more big, we change the scale factor of it to (40, 4.4, 40).
	Because we don't have any dynamic lights in the scene, we switch off the lighting,
	and we set the file terrain-texture.jpg as texture for the terrain and 
	detailmap3.jpg as second texture, called detail map. At last, we set
	the scale values for the texture: The first texture will be repeated only one time over 
	the whole terrain, and the second one (detail map) 20 times. 
	*/

	scene::ITerrainSceneNode* terrain = m_EditorManager->getSceneMngr()->addTerrainSceneNode(
		"media/Terrain/terrain-heightmap.bmp",
		0,												// parent node
		-1,												// node id
		core::vector3df(-3000.f, -440.f, -3000.f),		// position
		core::vector3df(0.f, 0.f, 0.f),					// rotation
		core::vector3df(40.f, 4.4f, 40.f),				// scale
		video::SColor ( 255, 255, 255, 255 ),			// vertexColor,
		3,												// maxLOD
		scene::ETPS_17,									// patchSize
		4												// smoothFactor
		);

	terrain->setMaterialFlag(EMF_LIGHTING, false);
	terrain->setMaterialTexture(0, m_EditorManager->getDriver()->getTexture("media/Terrain/terrain-texture.jpg"));
	terrain->setMaterialTexture(1, m_EditorManager->getDriver()->getTexture("media/Terrain/detailmap2.jpg"));
	terrain->setMaterialType(EMT_DETAIL_MAP);
	terrain->scaleTexture(1.0f, 20.0f);

	/*
	To be able to do collision with the terrain, we create a triangle selector.
	If you want to know what triangle selectors do, just take a look into the 
	collision tutorial. The terrain triangle selector works together with the
	terrain. To demonstrate this, we create a collision response animator 
	and attach it to the camera, so that the camera will not be able to fly 
	through the terrain.
	*/

	// create triangle selector for the terrain	
	scene::ITriangleSelector* selector = m_EditorManager->getSceneMngr()->createTerrainTriangleSelector(terrain, 0);
	terrain->setTriangleSelector(selector);
	selector->drop();

	// add irrlicht logo
	m_EditorManager->getGUIEnvironment()->addImage(m_EditorManager->getDriver()->getTexture("media/irrlichtlogo2.png"), core::position2d<s32>(770,580));

	return true;
}

/**
 * \brief Adds triangle selector to the terrain (used on reloading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CEditorLevel::addTerrainSelector()
{
	ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_EditorManager->getSceneMngr()->getSceneNodeFromType(ESNT_TERRAIN);
	if (terrain)
	{
		scene::ITriangleSelector* selector = m_EditorManager->getSceneMngr()->createTerrainTriangleSelector(terrain, 0);
		terrain->setTriangleSelector(selector);
		selector->drop();
	}
}

/**
 * \brief This function will become more complex when multiselect gets functional
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorLevel::SingleObjectSelected()
{
	return (m_SelectedGameObject != NULL);
}

void CEditorLevel::OnLoadMap(stringc filename)
{
	SetMapName(filename);
	m_EditorManager->m_ID = 0; //cleaning old map
	m_SelectedGameObject = NULL;
	m_PreviewGameObject = 0;
	m_EditorManager->getSceneMngr()->clear();
	if (filename.find(".irr") > -1)
	{
		if (m_EditorManager->getSceneMngr()->loadScene(filename.c_str(),this))
		{
			m_EditorManager->getSceneMngr()->getActiveCamera()->remove();
			CreateCamera(vector3df(100,300,100));
			addTerrainSelector();
		}
	}
	else
	{
		stringw message  = "Could not load map! Make sure you pick file with .irr extension.";
		m_EditorManager->getGUIEnvironment()->addMessageBox(L"Error Loading", message.c_str());
	}
}

/**
 * \brief Mouse and Keyboard events are handled here. 
 * Every mouse click on game object is caught, mouse move will move the 
 * models around, del key will delete them from the map... etc.
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CEditorLevel::OnEvent(const SEvent& event)
{
	if(event.EventType == EET_KEY_INPUT_EVENT)
	{
		if(event.KeyInput.Key == KEY_SHIFT)
		{
			m_bShiftPressed = event.KeyInput.PressedDown;
		}
		//Delete selected game object
		if ((event.KeyInput.PressedDown) && (event.KeyInput.Key == KEY_DELETE) && (m_SelectedGameObject))
		{
			m_ListOfGameObjects.erase(m_SelectedGameObject->getID(),1);
			//Shift all SceneNodes ID Numbers down by 1
			for (u32 i = m_SelectedGameObject->getID(); i < m_ListOfGameObjects.size(); i++)
			{
				m_EditorManager->getSceneMngr()->getSceneNodeFromId(i+1)->setID(i);
			}
			//delete the bastard
			m_SelectedGameObject->remove();
			m_SelectedGameObject = NULL;
			m_EditorManager->m_ID--;
			m_EditorManager->getGUIManager()->ClearProperties();
		}
	}

	if(event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		switch(event.MouseInput.Event)
		{
			case EMIE_LMOUSE_PRESSED_DOWN:
				{
					if (m_SelectedGameObject && m_bShiftPressed)
					{
						//we have to save first X because rotation jerks on start, have no clue why
						m_NodeRotationX = event.MouseInput.X; 
						//we have to include any already existing rotation of object
						m_CurrentRotationOfSelectedNode = m_SelectedGameObject->getRotation(); 
						//lets do the rotation! (see EMIE_MOUSE_MOVED below)
						m_bRotateSelectedNode = true;
					}
					else
					{
						m_bRotateSelectedNode = false;
						triangle3df instersection_triangle;
						ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_EditorManager->getSceneMngr()->getSceneNodeFromType(ESNT_TERRAIN);
						if(terrain)
						{
							ITriangleSelector* selector = terrain->getTriangleSelector();
							line3d<f32> picking_line = m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getRayFromScreenCoordinates(m_EditorManager->getDevice()->getCursorControl()->getPosition());
							m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getCollisionPoint(picking_line,selector,m_instersection_point,instersection_triangle);
							m_SelectedGameObject = m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getSceneNodeFromScreenCoordinatesBB(m_EditorManager->getDevice()->getCursorControl()->getPosition());
							if(m_SelectedGameObject && m_SelectedGameObject->getType() != ESNT_TERRAIN)
							{
								//object is selected on click, and its position is memorized
								//object is moved on event mouse move.
								m_bMoveSelectedNode = true;
								m_CurrentZoom = m_SelectedGameObject->getScale();
								m_selectedGameObjectStartMovingPosition = m_SelectedGameObject->getPosition();

								//tell gui to change properties to this selected item
								TGameObjectProperty properties;
								//Format position string out of vector
								stringw temp_position = stringw(m_SelectedGameObject->getPosition().X) + " " + stringw(m_SelectedGameObject->getPosition().Y) + " " + stringw(m_SelectedGameObject->getPosition().Z);
								properties.name = m_SelectedGameObject->getName();
								m_ListOfGameObjects[m_SelectedGameObject->getID()]->name = m_SelectedGameObject->getName();
								IAnimatedMeshSceneNode* m = (IAnimatedMeshSceneNode*)m_SelectedGameObject;
								properties.mesh = m_EditorManager->getSceneMngr()->getMeshCache()->getMeshFilename(m->getMesh());
								m_ListOfGameObjects[m_SelectedGameObject->getID()]->mesh = properties.mesh;
								properties.id = m_SelectedGameObject->getID();
								properties.isContainer = m_ListOfGameObjects[m_SelectedGameObject->getID()]->isContainer;
								properties.isPickable = m_ListOfGameObjects[m_SelectedGameObject->getID()]->isPickable;
								properties.isTrigger = m_ListOfGameObjects[m_SelectedGameObject->getID()]->isTrigger;
								properties.isNPC = m_ListOfGameObjects[m_SelectedGameObject->getID()]->isNPC;
								properties.isMonster = m_ListOfGameObjects[m_SelectedGameObject->getID()]->isMonster;
								properties.isVisible = m_SelectedGameObject->isVisible();
								properties.position = temp_position;
								properties.description = m_ListOfGameObjects[m_SelectedGameObject->getID()]->description.c_str();
								properties.script = m_ListOfGameObjects[m_SelectedGameObject->getID()]->script.c_str();
								m_EditorManager->getGUIManager()->SetProperties(properties);
							}
							else 
							{
								m_SelectedGameObject = NULL;
								m_EditorManager->getGUIManager()->ClearProperties();
							}
						}
					}
				}
				break;
			case EMIE_LMOUSE_LEFT_UP:
				m_bMoveSelectedNode = false;
				m_bRotateSelectedNode = false;
				m_bElementAtHand = false;
				break;
			case EMIE_MOUSE_MOVED:
				//Rotate selected node
				if (m_bRotateSelectedNode)
				{
					//Moving mouse along X axis will cause rotation of model around Y axis (don't let this confuse you)
					f32 RotX = (f32) (m_NodeRotationX - event.MouseInput.X);
					m_SelectedGameObject->setRotation(m_CurrentRotationOfSelectedNode + vector3df(0,RotX,0));
				}
				//Move selected node
				if (m_bMoveSelectedNode)
				{
					if(m_SelectedGameObject)
					{
						//TGameObjectProperty properties;
						vector3df instersection_point;
						triangle3df instersection_triangle;
						ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_EditorManager->getSceneMngr()->getSceneNodeFromType(ESNT_TERRAIN);
						if (terrain)
						{
							ITriangleSelector* selector = terrain->getTriangleSelector();
							line3d<f32> picking_line = m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getRayFromScreenCoordinates(m_EditorManager->getDevice()->getCursorControl()->getPosition());
							m_EditorManager->getSceneMngr()->getSceneCollisionManager()->getCollisionPoint(picking_line,selector,instersection_point,instersection_triangle);
							m_SelectedGameObject->setPosition(m_selectedGameObjectStartMovingPosition + instersection_point - m_instersection_point);
							//set properties to reflect position coord change while moving object
							stringw temp_position = stringw(m_SelectedGameObject->getPosition().X) + " " + stringw(m_SelectedGameObject->getPosition().Y) + " " + stringw(m_SelectedGameObject->getPosition().Z);
							m_EditorManager->getGUIManager()->SetPosition(temp_position);
						}
					}
				}
				break;
			case EMIE_MOUSE_WHEEL:
				{
					//Scale selected object with SHIFT + mouse wheel
					if (m_SelectedGameObject && m_bShiftPressed)
					{
						if (event.MouseInput.Wheel > 0)
							m_CurrentZoom += vector3df(0.1f,0.1f,0.1f);
						else
							m_CurrentZoom -= vector3df(0.1f,0.1f,0.1f);
						m_SelectedGameObject->setScale(m_CurrentZoom);
					}
				}
				break;
			default:
				break;
		}
	}
	return false;
}