/** 
 * \file GameGUI.cpp
 * \brief GUI class creates menus and property windows and buttons and handles all the user clicks.
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "GameGUI.h"
#include "../level/LevelManager.h"
#include "../GameManager.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CGameGUI::CGameGUI()
{
	m_GameManager = NULL;
	m_wnd_containerContent = NULL;
	m_ContainerID = -1;
	m_ContainerTakeAllButton = NULL;
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CGameGUI::~CGameGUI()
{
}

/**
 * \brief Event handler - handles user clicks (mouse and keyboard).
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CGameGUI::OnEvent(const SEvent& event)
{
	if(!m_GameManager)
		return false;

	IGUIEnvironment* env = m_GameManager->getGUIEnvironment();

	if(event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		core::position2d<s32> p(event.MouseInput.X, event.MouseInput.Y);

		//This is code for silly item moving from inventory to container and back (on click)
		if ((event.MouseInput.Event == EMIE_LMOUSE_PRESSED_DOWN) || (event.MouseInput.Event == EMIE_LMOUSE_LEFT_UP))
		{
			if (m_wnd_containerContent)
			{
				//User has opened chest!
				if (event.MouseInput.Event == EMIE_LMOUSE_PRESSED_DOWN) //you'd think this is an error, but no.
				{
					IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(GAME_CONST_CONTAINER_ID);
					for (u32 i = 0; i < container->GetNumItems(); i++)
					{
						if (container->isInsideSlot(i,p))
						{
							//Simple mouse click is handled like request to move item.
							//Advanced implementation would involve selection of item
							//Then pressing del key would remove item 
							//Dragging it with mouse away from component, etc
							CPickableItem* item = container->GetItem(i);
							if(item != NULL)
							{
								if(!m_Inventory->isFull())
								{
									m_Inventory->InsertItem(item); //add to inventory
									container->RemoveItem(i);
									return true;
								}
								else
								{
									//shib msg: inventory full!
									stringw message = "Can't add any more items to inventory!";
									env->addMessageBox(L"Inventory Full", message.c_str());
									return true;
								}
							}
						}
					}
					//new for loop for inventory
					for (u32 i = 0; i < m_Inventory->GetNumItems(); i++)
					{
						if (m_Inventory->isInsideSlot(i,p))
						{
							//Simple mouse click is handled like request to move item.
							//Advanced implementation would involve selection of item
							//Then pressing del key would remove item 
							//Dragging it with mouse away from component, etc
							CPickableItem* item = m_Inventory->GetItem(i);
							if(item != NULL)
							{
								if(!container->isFull())
								{
									container->InsertItem(item); //add to inventory
									m_Inventory->RemoveItem(i);
									return true;
								}
								else
								{
									//shib msg: inventory full!
									stringw message = "Can't add any more items to inventory!";
									env->addMessageBox(L"Inventory Full", message.c_str());
									return true;
								}
							}
						}
					}
				}

				//should return true if (m_wnd_containerContent), even if doing nothing with icons, so that
				//mouse events are ignored (in rest of the program code) while Container is displayed
				return true; 
			}
		}
	}

	//GUI event is special kind of event (not simple mouse or keyboard) generated by irrlicht engine
	//it simplifies event handling by telling us directly if button was clicked or window was moved 
	//or menu item was selected... etc.
	if (event.EventType == EET_GUI_EVENT)
		{
			s32 id = event.GUIEvent.Caller->getID();

			switch(event.GUIEvent.EventType)
			{
			case EGET_ELEMENT_CLOSED:
				{
					if(event.GUIEvent.Caller == m_wnd_containerContent)
					{
						//Game object should memorize new container content on close.
						IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(GAME_CONST_CONTAINER_ID);
						CPickableItem* item;
						
						//remove existing container content
						m_GameManager->getLevelManager()->m_pFirstTutorialLevel->m_ListOfGameObjects[m_ContainerID]->ClearPickableItems();

						//assign new content (if any) to game object
						for ( u32 index = 0; index < container->GetNumItems(); index++)
						{
							item = container->GetItem(index);
							if (item != NULL)
							{
								m_GameManager->getLevelManager()->m_pFirstTutorialLevel->m_ListOfGameObjects[m_ContainerID]->AddPickableItem(item->getName().c_str());
							}
						}

						m_wnd_containerContent = NULL;
					}
				}
				break;

			case EGET_BUTTON_CLICKED:
				{
					//Container button Take All.
					if(m_ContainerTakeAllButton->getID() == event.GUIEvent.Caller->getID())
					{
						//Save Container Content and close the window
						IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(GAME_CONST_CONTAINER_ID);

						CPickableItem* item;
						
						for ( u32 index = 0; index < container->GetNumItems(); index++)
						{
							item = container->GetItem(index);
							if (item != NULL)
							{
								if (!m_Inventory->isFull())
								{
									m_Inventory->InsertItem(item); //add to inventory
									container->RemoveItem(index); //remove from container
								}
								else
								{
									//shib msg: inventory full!
									stringw message = "Can't add any more items to inventory!";
									env->addMessageBox(L"Inventory Full", message.c_str());
									break;
								}
							}
						}

						return true;
					}
				}
				break;
			}
		}
	
	return false;
}

/**
 * \brief Finds Icon file and loads 2d image to container object.
 * This function is called when user wants to add content to container.
 * When he clicks on Item, icon of that Item is moved to the container.
 * This can be used later for draging icons around to other containers.
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 
void CGameGUI::SetIconAtHand(stringw parent, stringw name)
{
	bool icon_found = false;
	stringc dir = "media/Icons/";
	//find file in the filesystem
	IFileSystem* fs = m_GameManager->getDevice()->getFileSystem();
	IFileList* filelist = NULL;
	fs->changeWorkingDirectoryTo(dir.c_str());
	filelist = fs->createFileList();
	stringc temp_filename;
	//we go through list of all files in given Folder
	for ( u32 a = 2; a < filelist->getFileCount(); a++) //start from nuber 2 because 0 and 1 are directories "." and ".."
	{
		if(!filelist->isDirectory(a))
		{
			temp_filename = filelist->getFileName(a);
			//check every file if containst our icon name
			if (temp_filename.find(name.c_str()) != -1)
			{
				//but accept only image files (add more later...)
				if (temp_filename.find(".jpg") + temp_filename.find(".bmp") + temp_filename.find(".png") > -3) 
				{
					icon_found = true;
					break;
				}
			}
		}
	}
	fs->changeWorkingDirectoryTo("../../");

	if(icon_found)
	{
		//Add Icon to container
		CPickableItem* item = new CPickableItem();
		dir += temp_filename.c_str();
		item->setName(dir.c_str());
		item->m_IconTexture = m_GameManager->getGUIEnvironment()->getVideoDriver()->getTexture(dir.c_str());
		IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(1357);
		container->InsertItem(item);
	}
	else
	{
		//shibMessage("Unfortunately, this icon is not found");
		stringw message = "File ";
		message += name.c_str();
		message += " is missing from folder media/Icons";
		m_GameManager->getGUIEnvironment()->addMessageBox(L"File is missing!", message.c_str());
	}

	filelist->drop();
}*/

/**
 * \brief Displays GUI Window with content of container object with given id
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameGUI::DisplayContainerContent(s32 id, 	IVideoDriver* driver, IGUIEnvironment* env, CLevelManager* levelManager)
{
	m_wnd_containerContent = env->addWindow(core::rect<s32>(250,100,470,380), false, L"Container Content", 0, 4500);
	m_ContainerTakeAllButton = env->addButton(core::rect<s32>(70,240,140,260), m_wnd_containerContent, 1355, L"Take All");

	ITexture* slotTex = driver->getTexture("media/Icons/slot.png");
	IGUIContainer* container = AddGUIContainer(
		env,
		core::rect<s32>(10,25,205,220), 
		m_wnd_containerContent, 
		GAME_CONST_CONTAINER_ID,
		core::dimension2d<s32>(3,3),
		core::dimension2d<s32>(1,1),
		slotTex
		);

	s32 numberOfItems = levelManager->m_pFirstTutorialLevel->m_ListOfGameObjects[id]->GetNumberOfPickableItems();
	for (s32 a=0; a< numberOfItems; a++)
	{
		stringc iconPath = levelManager->m_pFirstTutorialLevel->m_ListOfGameObjects[id]->GetPickableItem(a);
		CPickableItem* item = new CPickableItem();
		item->setName(iconPath.c_str());
		item->m_IconTexture = driver->getTexture(iconPath.c_str());
		container->InsertItem(item);
	}

	m_ContainerID = id;
}

/**
 * \brief Initializes all windows and gui elements...
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CGameGUI::Init(CGameManager* gameMngr)
{
	m_GameManager = gameMngr;

	//Setting GUI transparency (alpha)
	for (s32 i=0; i<irr::gui::EGDC_COUNT ; ++i)
	{
		video::SColor col = m_GameManager->getGUIEnvironment()->getSkin()->getColor((EGUI_DEFAULT_COLOR)i);
		col.setAlpha(200);
		m_GameManager->getGUIEnvironment()->getSkin()->setColor((EGUI_DEFAULT_COLOR)i, col);
	}

	//Add Inventory GUI
	ITexture* slotTex = m_GameManager->getDriver()->getTexture("media/Icons/slot.png");
	m_Inventory = AddGUIContainer(
		m_GameManager->getGUIEnvironment(),
		core::rect<s32>(115,520,635,585),
		0,
		GAME_CONST_INVENTORY_ID,
		core::dimension2d<s32>(8,1),
		core::dimension2d<s32>(1,1),
		slotTex
		);

	printf("Game GUI Initialized");

	return true;
}