/** 
 * \file LevelManager.cpp
 * \brief File LevelManager.cpp takes care of all game levels.
 *
 * takes care of saving/loading levels
 * manages list of levels
 * tightly communicates with GUI environment. 
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "LevelManager.h"
#include "../GameManager.h"
#include "../gui/GameGUI.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CLevelManager::CLevelManager()
{
	m_GameManager = NULL;
	m_bShiftPressed = false;
	m_CurrentZoom = vector3df(1.0f,1.0f,1.0f);
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CLevelManager::~CLevelManager()
{
}

/**
 * \brief Anything special (not belonging to irrlicht scene) is rendered here
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevelManager::OnRender()
{
}

/**
 * \brief Init function stores pointer to CGameManager and creates initial level.
 * \author Petar Bajic
 * \date July, 21 2008.
 */
bool CLevelManager::Init(CGameManager* gameMngr, CGameGUI* gameGUI)
{
	m_GameManager = gameMngr;

	m_pFirstTutorialLevel = new CLevel();
	m_pFirstTutorialLevel->Init(m_GameManager->getDevice(),m_GameManager->getFS(),m_GameManager->getDriver(),m_GameManager->getSceneMngr());

	//what is this for?
	m_GameManager->getDriver()->setTextureCreationFlag(video::ETCF_ALWAYS_32_BIT, true);

	// add irrlicht logo
	m_GameManager->getGUIEnvironment()->addImage(m_GameManager->getDriver()->getTexture("media/irrlichtlogo2.png"), core::position2d<s32>(50,50));

	m_GameGUI = gameGUI;
	return true;
}

/**
 * \brief Creates camera in initial position (used for loading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevelManager::CreateCamera(vector3df position)
{
	RTSCamera* camera = new RTSCamera(m_GameManager->getDevice(),m_GameManager->getSceneMngr()->getRootSceneNode(),m_GameManager->getSceneMngr(),-1,100.0f,10.0f,100.0f);
	camera->setPosition(position);
}

/**
 * \brief Manages Map Loading
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CLevelManager::OnLoadMap(const c8* map_filename)
{
	if (m_pFirstTutorialLevel->Load(map_filename))
	{
		CreateCamera(vector3df(100,300,100));
		return true;
	}

	//Display error to console
	stringw message  = "Map: ";
	message += map_filename;
	message += " can not be loaded!";
	m_GameManager->getGUIEnvironment()->addMessageBox(L"Error Loading Map", message.c_str());

	return false;
}

/**
 * \brief Mouse and Keyboard events are handled here. 
 * Every mouse click on game object is caught, interaction with the map is handled:
 * doors lead to another map, containers display their content, NPCs start talking...
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CLevelManager::OnEvent(const SEvent& event)
{
	if(event.EventType == EET_KEY_INPUT_EVENT)
	{
		if(event.KeyInput.Key == KEY_SHIFT)
		{
			m_bShiftPressed = event.KeyInput.PressedDown;
		}
	}

	if(event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		switch(event.MouseInput.Event)
		{
			case EMIE_LMOUSE_PRESSED_DOWN:
				{
					//check if container is clicked on
					s32 id = m_pFirstTutorialLevel->OnMouseClick();
					if (id > -1)
					{
						m_GameGUI->DisplayContainerContent(id,
							m_GameManager->getDriver(), m_GameManager->getGUIEnvironment(), this);
						
						return true;
					}
				}
				break;
			case EMIE_LMOUSE_LEFT_UP:
				break;
			case EMIE_MOUSE_MOVED:
				{
				}
				break;
			case EMIE_MOUSE_WHEEL:
				{
				}
				break;
			default:
				break;
		}
	}
	return false;
}