/** 
 * \file GameGUI.cpp
 * \brief GUI class creates menus and property windows and buttons and handles all the user clicks.
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "GameGUI.h"
#include "../level/LevelManager.h"
#include "../GameManager.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CGameGUI::CGameGUI()
{
	m_GameManager = NULL;
	m_wnd_containerContent = NULL;
	m_wnd_options = 0;
	m_wnd_charSheet = 0;
	m_ContainerID = -1;
	m_ContainerTakeAllButton = NULL;
	m_bDraggingPickableItem = false;
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CGameGUI::~CGameGUI()
{
}

/**
 * \brief Event handler - handles user clicks (mouse and keyboard).
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CGameGUI::OnEvent(const SEvent& event)
{
	if(!m_GameManager)
		return false;

	IGUIEnvironment* env = m_GameManager->getGUIEnvironment();

	if(event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		core::position2d<s32> p(event.MouseInput.X, event.MouseInput.Y);

		//This is code for silly item moving from inventory to container and back (on click)
		if ((event.MouseInput.Event == EMIE_LMOUSE_PRESSED_DOWN) || (event.MouseInput.Event == EMIE_LMOUSE_LEFT_UP))
		{
			//if player clicked on settings button return, gui event will follow..
			if (m_SettingsButton->getAbsoluteClippingRect().isPointInside(p) || 
				m_CSheetButton->getAbsoluteClippingRect().isPointInside(p))
			{
				return true; //makes sure others dont handle this click, like player move
			}
			if (m_wnd_containerContent)
			{
				//User has opened chest!
				if (event.MouseInput.Event == EMIE_LMOUSE_PRESSED_DOWN) //you'd think this is an error, but no. thing is, we handle only mouse press down clicks in item trensfer between container and inventory, BUT we handle both mouse press up and down if container is open it returns true so mouse up and down events are both ignored in the scene below. :)
				{
					IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(GAME_CONST_CONTAINER_ID);
					for (u32 i = 0; i < container->GetNumItems(); i++)
					{
						if (container->isInsideSlot(i,p))
						{
							if(m_bDraggingPickableItem)
							{
								CPickableItem* item = container->GetItem(i);
								if(item == NULL)
								{
									container->InsertItem(m_pDraggedPickableItem, i); //add to inventory
									m_bDraggingPickableItem = false;
									m_pDraggedPickableItem = 0;
									return true;
								}
								else
								{
									//msg Cant combine two items in that container, do it in inventory
									AddConsoleText(L"Can't combine two items in that container.");
									return true;
								}
							}
							else
							{
								//Simple mouse click is handled like request to move item.
								//Advanced implementation would involve selection of item
								//Then pressing del key would remove item 
								//Dragging it with mouse away from component, etc
								CPickableItem* item = container->GetItem(i);
								if(item != NULL)
								{
									if(!m_Inventory->isFull())
									{
										m_Inventory->InsertItem(item); //add to inventory
										container->RemoveItem(i);
										return true;
									}
									else
									{
										//shib msg: inventory full!
										stringw message = "Can't add any more items to inventory!";
										env->addMessageBox(L"Inventory Full", message.c_str());
										return true;
									}
								}
							}
						}
					}
					//new for loop for inventory
					for (u32 i = 0; i < m_Inventory->GetNumItems(); i++)
					{
						if (m_Inventory->isInsideSlot(i,p))
						{
							if(m_bDraggingPickableItem)
							{
								CPickableItem* item = container->GetItem(i);
								if(item == NULL)
								{
									m_Inventory->InsertItem(m_pDraggedPickableItem, i); //add to inventory
									m_bDraggingPickableItem = false;
									m_pDraggedPickableItem = 0;
									return true;
								}
								else
								{
									//msg Can't combine these two items
									AddConsoleText(L"Can't combine these two items");
									return true;
								}
							}
							else
							{
								//Simple mouse click is handled like request to move item.
								//Advanced implementation would involve selection of item
								//Then pressing del key would remove item 
								//Dragging it with mouse away from component, etc
								CPickableItem* item = m_Inventory->GetItem(i);
								if(item != NULL)
								{
									if(!container->isFull())
									{
										container->InsertItem(item); //add to inventory
										m_Inventory->RemoveItem(i);
										return true;
									}
									else
									{
										//shib msg: inventory full!
										stringw message = "Can't add any more items to inventory!";
										env->addMessageBox(L"Inventory Full", message.c_str());
										return true;
									}
								}
							}
						}
					}
				}
				else if(event.MouseInput.Event == EMIE_RMOUSE_PRESSED_DOWN)
				{
					if(m_bDraggingPickableItem == false)
					{
						for (u32 i = 0; i < m_Inventory->GetNumItems(); i++)
						{
							if (m_Inventory->isInsideSlot(i,p))
							{
								//Right mouse click activate item under cursor
								m_pDraggedPickableItem = m_Inventory->GetItem(i);
								if(m_pDraggedPickableItem != 0)
								{
									//m_pDraggedPickableItem->index = i;
									//activate item script
									m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONUSE, m_pDraggedPickableItem->m_Script, 0);
									m_pDraggedPickableItem = 0;
								}
								return true;
							}
						}
					}
				}

				//should return true if (m_wnd_containerContent), even if doing nothing with icons, so that
				//mouse events are ignored (in rest of the program code) while Container is displayed
				return true; 
			}
			//container is not opened, check if clicked in inventory
			else if (event.MouseInput.Event == EMIE_LMOUSE_PRESSED_DOWN)
			{
				if(m_bDraggingPickableItem == false)
				{
					for (u32 i = 0; i < m_Inventory->GetNumItems(); i++)
					{
						if (m_Inventory->isInsideSlot(i,p))
						{
							//Simple mouse click moves item under cursor and item can be
							//moved around, dragged with mouse away from component, etc
							m_pDraggedPickableItem = m_Inventory->GetItem(i);
							if(m_pDraggedPickableItem != NULL)
							{
								//item being dragged
								m_bDraggingPickableItem = true;
								m_Inventory->RemoveItem(i);
								stringc name = m_pDraggedPickableItem->getName();
							}
							return true;
						}
					}
				}
				else
				{
					//Item is being dragged
					for (u32 i = 0; i < m_Inventory->GetNumItems(); i++)
					{
						if (m_Inventory->isInsideSlot(i,p))
						{
							//Simple mouse click moves item under cursor and item can be
							//moved around, dragged with mouse away from component, etc
							CPickableItem* item = m_Inventory->GetItem(i);
							if(item != NULL)
							{
								//item being droped on another item
								//if combinable combination
								//combine items
								//m_bItemDragged = false;
								//else
								AddConsoleText(L"Can't combine these two items");
								return true;
							}
							else
							{
								//item dropped in empty space in inventory
								m_Inventory->InsertItem(m_pDraggedPickableItem, i); //add to inventory
								m_bDraggingPickableItem = false;
								m_pDraggedPickableItem = 0;
								return true;
							}
						}
					}
				}
			}
		}
		else if(event.MouseInput.Event == EMIE_RMOUSE_PRESSED_DOWN)
		{
			if(m_bDraggingPickableItem == false)
			{
				for (u32 i = 0; i < m_Inventory->GetNumItems(); i++)
				{
					if (m_Inventory->isInsideSlot(i,p))
					{
						//Right mouse click activate item under cursor
						m_pDraggedPickableItem = m_Inventory->GetItem(i);
						if(m_pDraggedPickableItem != 0)
						{
							//m_pDraggedPickableItem->index = i;
							//activate item script
							m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONUSE, m_pDraggedPickableItem->m_Script, 0);
							m_pDraggedPickableItem = 0;
						}
						return true;
					}
				}
			}
		}
	}

	//GUI event is special kind of event (not simple mouse or keyboard) generated by irrlicht engine
	//it simplifies event handling by telling us directly if button was clicked or window was moved 
	//or menu item was selected... etc.
	if (event.EventType == EET_GUI_EVENT)
		{
			s32 id = event.GUIEvent.Caller->getID();

			switch(event.GUIEvent.EventType)
			{
			case EGET_ELEMENT_CLOSED:
				{
					if (event.GUIEvent.Caller == m_wnd_options)
					{
						m_wnd_options = 0;
					}
					if (event.GUIEvent.Caller == m_wnd_charSheet)
					{
						m_wnd_charSheet = 0;
					}
					else if(event.GUIEvent.Caller == m_wnd_containerContent)
					{
						//Game object should memorize new container content on close.
						IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(GAME_CONST_CONTAINER_ID);
						CPickableItem* item;
						
						//remove existing container content
						m_GameManager->getLevelManager()->m_pLevels[m_GameManager->getLevelManager()->m_LevelIndex]->m_ListOfGameObjects[m_ContainerID]->ClearPickableItems();

						//assign new content (if any) to game object
						for ( u32 index = 0; index < container->GetNumItems(); index++)
						{
							item = container->GetItem(index);
							if (item != NULL)
							{
								m_GameManager->getLevelManager()->m_pLevels[m_GameManager->getLevelManager()->m_LevelIndex]->m_ListOfGameObjects[m_ContainerID]->AddPickableItem(item->getName().c_str());
							}
						}

						m_wnd_containerContent = NULL;
					}
				}
				break;

			case EGET_BUTTON_CLICKED:
				{
					//Container button Take All.
					if(m_wnd_containerContent && (m_ContainerTakeAllButton->getID() == event.GUIEvent.Caller->getID()))
					{
						//Save Container Content and close the window
						IGUIContainer* container = (IGUIContainer*) m_wnd_containerContent->getElementFromId(GAME_CONST_CONTAINER_ID);

						CPickableItem* item;
						
						for ( u32 index = 0; index < container->GetNumItems(); index++)
						{
							item = container->GetItem(index);
							if (item != NULL)
							{
								if (!m_Inventory->isFull())
								{
									m_Inventory->InsertItem(item); //add to inventory
									container->RemoveItem(index); //remove from container
								}
								else
								{
									//shib msg: inventory full!
									stringw message = "Can't add any more items to inventory!";
									env->addMessageBox(L"Inventory Full", message.c_str());
									break;
								}
							}
						}

						return true;
					}
					if(!m_wnd_charSheet && !m_wnd_options && !m_wnd_containerContent && m_SettingsButton->getID() == event.GUIEvent.Caller->getID())
					{
						//bring up settings menu!
						DisplayOptionsWindow(env);
						return true;
					}
					if(!m_wnd_charSheet && !m_wnd_options && !m_wnd_containerContent && m_CSheetButton->getID() == event.GUIEvent.Caller->getID())
					{
						//bring up char sheet!
						DisplayCharacterSheetWindow(env);
						return true;
					}
					if(m_wnd_options && (m_ExitButton->getID() == event.GUIEvent.Caller->getID()))
					{
						//exit game
						m_GameManager->getDevice()->closeDevice();
						return true;
					}
				}
				break;
			}
		}
	
	return false;
}

void CGameGUI::RemovePickableFromInventory(s32 index)
{
	m_Inventory->RemoveItem(index);
}

void CGameGUI::AddPickableToInventory(stringw item_name)
{
	if (!m_Inventory->isFull())
	{
		CPickableItem* item = new CPickableItem();
		item->m_CoreName = item_name.c_str();
		stringw iconName;
		iconName += L"media/Icons/";
		iconName += item_name.c_str();
		iconName += L".png";
		item->setName(stringc(iconName.c_str()));
		item->m_IconTexture = m_GameManager->getGUIEnvironment()->getVideoDriver()->getTexture(stringc(iconName.c_str()).c_str());
		item->m_Script = item_name.c_str();
		item->m_Script += L".script";
		if (item->m_IconTexture != 0)
		{
			m_Inventory->InsertItem(item); //add to inventory
		}
		else
		{
			//msg: wrong icon texture name!
		}
	}
	else
	{
		//msg: Inventory is full
	}
}

/**
 * \brief Displays GUI Window with player abilities and skills
 * \author Petar Bajic
 * \date July, 21 2008.
 */
void CGameGUI::DisplayCharacterSheetWindow(IGUIEnvironment* env)
{
	m_wnd_charSheet = env->addWindow(core::rect<s32>(100,60,750,550), false, L"Character Sheet", 0, 6500);
	ITexture* texture = m_GameManager->getDriver()->getTexture("media/Icons/bar.png");
	IGUIStaticText* text_abilities = env->addStaticText(L"Abilities:",rect<s32>(20,34,100,60),false,false,m_wnd_charSheet);
	text_abilities->setOverrideFont(env->getBuiltInFont());
	text_abilities->setOverrideColor(SColor(255,0,140,0));
	IGUIStaticText* text_skills = env->addStaticText(L"Skills:",rect<s32>(20,250,100,270),false,false,m_wnd_charSheet);
	text_skills->setOverrideFont(env->getBuiltInFont());
	text_skills->setOverrideColor(SColor(255,0,140,0));

	//char image
	ITexture* charTex = env->getVideoDriver()->getTexture("media/Icons/char.png");
	IGUIImage* img2 = env->addImage(rect<s32>(300,50,600,450),m_wnd_charSheet);
	img2->setImage(charTex);

	//m_ListOfAbilities
	for(u32 i=0; i<m_GameManager->getPC()->m_ListOfAbilities.size(); i++)
	{
		cs_ability_bars[i] = new CGUIBar(120, 120+i*30, 220, 140+i*30, 
			m_GameManager->getPC()->m_ListOfAbilities[i].max, 
			m_GameManager->getPC()->m_ListOfAbilities[i].value, 
			texture, 
			m_GameManager->getPC()->m_ListOfAbilities[i].name.c_str());
	}

	//m_ListOfSkills
	for(i=0; i<m_GameManager->getPC()->m_ListOfSkills.size(); i++)
	{
		cs_skill_bars[i] = new CGUIBar(120, 330+i*30, 220, 350+i*30,
			m_GameManager->getPC()->m_ListOfSkills[i].max, 
			m_GameManager->getPC()->m_ListOfSkills[i].value, 
			texture, 
			m_GameManager->getPC()->m_ListOfSkills[i].name.c_str());
	}
}

/**
 * \brief Displays GUI Window with options
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameGUI::DisplayOptionsWindow(IGUIEnvironment* env)
{
		m_wnd_options = env->addWindow(core::rect<s32>(250,100,470,380), false, L"Options", 0, 5500);
		IGUIButton* but = env->addButton(core::rect<s32>(70,40,140,60), m_wnd_options, 5501, L"Save Game");
		but->setEnabled(false);
		but = env->addButton(core::rect<s32>(70,80,140,100), m_wnd_options, 5502, L"Load Game");
		but->setEnabled(false);
		but = env->addButton(core::rect<s32>(70,120,140,140), m_wnd_options, 5503, L"Settings");
		but->setEnabled(false);
		m_ExitButton = env->addButton(core::rect<s32>(70,160,140,180), m_wnd_options, 5504, L"Exit Game");
}

/**
 * \brief Displays GUI Window with content of container object with given id
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameGUI::DisplayContainerContent(s32 id,	IVideoDriver* driver, IGUIEnvironment* env, CLevelManager* levelManager)
{
	m_wnd_containerContent = env->addWindow(core::rect<s32>(250,100,470,380), false, L"Container Content", 0, 4500);
	m_ContainerTakeAllButton = env->addButton(core::rect<s32>(70,240,140,260), m_wnd_containerContent, 1355, L"Take All");

	ITexture* slotTex = driver->getTexture("media/Icons/slot.png");
	IGUIContainer* container = AddGUIContainer(
		env,
		core::rect<s32>(10,25,165,180), 
		m_wnd_containerContent, 
		GAME_CONST_CONTAINER_ID,
		core::dimension2d<s32>(3,3),
		core::dimension2d<s32>(1,1),
		slotTex
		);

	s32 numberOfItems = levelManager->m_pLevels[levelManager->m_LevelIndex]->m_ListOfGameObjects[id]->GetNumberOfPickableItems();
	for (s32 a=0; a< numberOfItems; a++)
	{
		stringc iconPath = levelManager->m_pLevels[levelManager->m_LevelIndex]->m_ListOfGameObjects[id]->GetPickableItem(a);
		stringw coreName= m_GameManager->getRootNameFromPathName(iconPath.c_str());
		CPickableItem* item = new CPickableItem();
		item->setName(iconPath.c_str());
		item->m_IconTexture = driver->getTexture(iconPath.c_str());
		item->m_CoreName = coreName.c_str();
		item->m_Script = coreName.c_str();
		item->m_Script += L".script";
		if (item->m_IconTexture == 0)
		{
			//icon file is missing!! critical error
			m_GameManager->getGUIEnvironment()->addMessageBox(L"File missing!", L"Item icon file is missing or relocated! Serious game error. Disengage! Disengage!!");
		}
		else
		{
			container->InsertItem(item);
		}
	}

	m_ContainerID = id;
}


/**
 * \brief Initializes all windows and gui elements...
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameGUI::AddConsoleText(stringw text)
{
	stringw temp;
	temp = m_Console->getText();
	temp += L"\n";
	temp += text.c_str();
	m_Console->setText(temp.c_str());
}

/**
 * \brief Initializes all windows and gui elements...
 * 
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CGameGUI::Init(CGameManager* gameMngr)
{
	m_GameManager = gameMngr;

	//init bars
	for (int i=0; i<10; i++)
	{
		cs_skill_bars[i] = 0;
		cs_ability_bars[i] = 0;
	}
	//Setting GUI transparency (alpha)
	for (s32 i=0; i<irr::gui::EGDC_COUNT ; ++i)
	{
		video::SColor col = m_GameManager->getGUIEnvironment()->getSkin()->getColor((EGUI_DEFAULT_COLOR)i);
		col.setAlpha(200);
		m_GameManager->getGUIEnvironment()->getSkin()->setColor((EGUI_DEFAULT_COLOR)i, col);
	}

	//Add Inventory GUI
	ITexture* slotTex = m_GameManager->getDriver()->getTexture("media/Icons/slot.png");
	m_Inventory = AddGUIContainer(
		m_GameManager->getGUIEnvironment(),
		core::rect<s32>(65,570,585,634),
		0,
		GAME_CONST_INVENTORY_ID,
		core::dimension2d<s32>(8,1),
		core::dimension2d<s32>(1,1),
		slotTex
		);

	//add 'console'
	m_Console = m_GameManager->getGUIEnvironment()->addEditBox(L"", core::rect<s32>(585,570,750,634), true, 0, 4519);
	m_Console->setMultiLine(true);
	m_Console->setAutoScroll(true);
	m_Console->setTextAlignment(EGUIA_UPPERLEFT,EGUIA_LOWERRIGHT);
	m_Console->setText(L"Game Initialized.");

	//add settings button:
	m_SettingsButton = m_GameManager->getGUIEnvironment()->addButton(rect<s32>(0,570,65,634),0,5600,L"");
	m_SettingsButton->setImage(m_GameManager->getDriver()->getTexture("media/Icons/house.png"));
	m_SettingsButton->setUseAlphaChannel(true);
	m_SettingsButton->setDrawBorder(false);

	//add character sheet button:
	m_CSheetButton = m_GameManager->getGUIEnvironment()->addButton(rect<s32>(750,570,815,634),0,6600,L"");
	m_CSheetButton->setImage(m_GameManager->getDriver()->getTexture("media/Icons/sheet.png"));
	m_CSheetButton->setUseAlphaChannel(true);
	m_CSheetButton->setDrawBorder(false);

	//add health bar
	ITexture* texture = 0;//m_GameManager->getDriver()->getTexture("media/Icons/bar.png");
	healthBar = new CGUIBar(40, 30, 200, 50, 20, 20, texture, L"Health");
	//m_GameManager->getGUIEnvironment()->addStaticText(L"Health",rect<s32>(10,30,40,50),false,false,0,-1,true);

	printf("Game GUI Initialized");

	return true;
}