/** 
 * \file Level.cpp
 * \brief File Level.cpp takes care of level structure - it hold together all game objects on the level.
 *
 * takes care of saving/loading level
 * manages game object arrays (adding deleting game objects)
 * translates events to game objects -> on mouse click, on rotate etc..
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "Level.h"
	

CLevel::CLevel()
{
	m_ID = 0;
}

CLevel::~CLevel()
{
}

bool CLevel::Init(IrrlichtDevice* device, IFileSystem* fs, IVideoDriver* driver, ISceneManager* smgr)
{
	m_ID = 0;
	m_pFS = fs;
	m_pDriver = driver;
	m_SMGR = smgr;
	m_pDevice = device;
	return true;
}

bool CLevel::Load(const c8* map_filename)
{
	m_ID = 0; //cleaning old map
	m_SMGR->clear();
	stringc currentDir = m_pFS->getWorkingDirectory();
	m_FullMapName = map_filename;
	if (m_SMGR->loadScene(map_filename,this))
	{
		m_SMGR->getActiveCamera()->remove(); //TODO: Make RTS camera load properly from irr file
		addTerrainSelector();
		return true;
	}

	return false;
}

/**
 * \brief Adds triangle selector to the terrain (used on reloading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevel::addTerrainSelector()
{
	ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_SMGR->getSceneNodeFromType(ESNT_TERRAIN);
	scene::ITriangleSelector* selector = m_SMGR->createTerrainTriangleSelector(terrain, 0);
	terrain->setTriangleSelector(selector);
	selector->drop();
}

IAttributes* CLevel::createUserData (irr::scene::ISceneNode *sceneNode)
{
	IAttributes* attr = m_pFS->createEmptyAttributes(m_pDriver);

	/*attr->addString("Homepage",L"www.mystic-peanut.com");

	if(sceneNode->getType() != ESNT_TERRAIN)
	{
		if((sceneNode->getID() >= 0) && (sceneNode->getID() < (s32)m_ListOfGameObjects.size()))
		{
			if (m_ListOfGameObjects[sceneNode->getID()]->isContainer)
				attr->addBool("isContainer",true);
		}
	}*/
	return attr;
}

void CLevel::OnReadUserData (irr::scene::ISceneNode *sceneNode, irr::io::IAttributes *userData)
{
	//save working dir cause we're about to change thigs
	stringc currentDir = m_pFS->getWorkingDirectory();
	
	//First, lets make string acrobatics.
	//We need path to dir where map is.
	//we need map name, just name, no path or extension, for xml container content
	//all we got is path to the map, but not from root, but from current working dir (we hope)
	s32 index = m_FullMapName.findLast('/');

	if (index < m_FullMapName.findLast('\\'))
		index = m_FullMapName.findLast('\\');
	
	if(index>-1)
	{
		//calculate full map path
		m_LoadingDir = currentDir;
		m_LoadingDir += "\\";
		m_LoadingDir += m_FullMapName.subString(0,index);
		m_LoadingDir.replace('/','\\');

		//calculate map name without path or extension
		m_MapName = m_FullMapName.subString(index+1, m_FullMapName.size()- index - 5);
	}

	m_pFS->changeWorkingDirectoryTo(m_LoadingDir.c_str());

	//ignoring camera, terrain and other special nodes with ID = -1
	if(sceneNode->getID() >= 0)
	{
		//adding new game object to the map
		
		CGameObject* gameObject = new CGameObject();
		if(userData->getAttributeAsBool("isContainer"))
		{
			gameObject->isContainer = true;

			//Load Existing Container Content (if any)
			stringc container_filename = "CC_";
			container_filename += m_MapName.c_str();
			container_filename += "_";
			container_filename += sceneNode->getID();
			container_filename += ".xml";
			io::IXMLReader* xml = m_pFS->createXMLReader(container_filename.c_str());
			while(xml && xml->read())
			{
				//Load container content from xml file to GameObjects list of pickable items
				if (core::stringw("Pickables") == xml->getNodeName())
				{
					//Add this element
					stringc item = xml->getAttributeValue(L"iconname");
					gameObject->AddPickableItem(item);
				}
			}
		
			if (xml)
				xml->drop(); // don't forget to delete the xml reader
		}
		if(userData->getAttributeAsBool("isPickable"))
		{
			gameObject->isPickable = true;
			gameObject->script = userData->getAttributeAsStringW("Script");
		}
		if(userData->getAttributeAsBool("isTrigger"))
		{
			gameObject->isTrigger = true;
		}
		gameObject->script = userData->getAttributeAsStringW("Script");
		gameObject->state = userData->getAttributeAsStringW("State");
		gameObject->id = m_ID;
		m_ListOfGameObjects.push_back(gameObject);
		if (m_ID != sceneNode->getID())
		{
			sceneNode->setID(m_ID); //This should never happen
			stringw message = "Node IDs are messed up. Map might not work properly.";
			//m_GameManager->getGUIEnvironment()->addMessageBox(L"Error loading map!", message.c_str());
		}

		m_ID++;
	}

	m_pFS->changeWorkingDirectoryTo(currentDir.c_str());
}

s32 CLevel::OnMouseClick()
{
	//triangle3df instersection_triangle;
	//vector3df instersection_point;
	//ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_SMGR->getSceneNodeFromType(ESNT_TERRAIN);
	//if(terrain)
	//{
		//ITriangleSelector* selector = terrain->getTriangleSelector();
		//line3d<f32> picking_line = m_SMGR->getSceneCollisionManager()->getRayFromScreenCoordinates(m_pDevice->getCursorControl()->getPosition());
		//m_SMGR->getSceneCollisionManager()->getCollisionPoint(picking_line,selector,instersection_point,instersection_triangle);
		ISceneNode* selectedGameObject = m_SMGR->getSceneCollisionManager()->getSceneNodeFromScreenCoordinatesBB(m_pDevice->getCursorControl()->getPosition());
		if(selectedGameObject && selectedGameObject->getType() != ESNT_TERRAIN)
		{
			return selectedGameObject->getID();
		}
	//}

	return -1;

}