/** 
 * \file LevelManager.cpp
 * \brief File LevelManager.cpp takes care of all game levels.
 *
 * takes care of saving/loading levels
 * manages list of levels
 * tightly communicates with GUI environment. 
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "LevelManager.h"
#include "../GameManager.h"
#include "../gui/GameGUI.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CLevelManager::CLevelManager()
{
	m_GameManager = NULL;
	m_bShiftPressed = false;
	m_bHoverOverActionItem = false;
	m_CurrentZoom = vector3df(1.0f,1.0f,1.0f);
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CLevelManager::~CLevelManager()
{
}

/**
 * \brief Anything special (not belonging to irrlicht scene) is rendered here
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevelManager::OnRender()
{
	//m_pLevels[m_LevelIndex]->m_SMGR->drawAll();
}

/**
 * \brief Init function stores pointer to CGameManager and creates initial level.
 * \author Petar Bajic
 * \date July, 21 2008.
 */
bool CLevelManager::Init(CGameManager* gameMngr, CGameGUI* gameGUI)
{
	m_GameManager = gameMngr;

	m_LevelIndex = -1;

	//what is this for?
	m_GameManager->getDriver()->setTextureCreationFlag(video::ETCF_ALWAYS_32_BIT, true);

	// add irrlicht logo
	m_GameManager->getGUIEnvironment()->addImage(m_GameManager->getDriver()->getTexture("media/irrlichtlogo2.png"), core::position2d<s32>(50,50));

	m_GameGUI = gameGUI;
	return true;
}

/**
 * \brief Creates camera in initial position (used for loading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevelManager::CreateCamera(vector3df position)
{
	//RTSCamera* camera = new RTSCamera(m_GameManager->getDevice(),m_GameManager->getSceneMngr()->getRootSceneNode(),m_GameManager->getSceneMngr(),-1,100.0f,10.0f,100.0f);
	RTSCamera* camera = new RTSCamera(m_GameManager->getDevice(),m_pLevels[m_LevelIndex]->m_SMGR->getRootSceneNode(),m_GameManager->getSceneMngr(),-1,100.0f,10.0f,100.0f);
	camera->setPosition(position);
}

/**
 * \brief Manages Map Loading
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CLevelManager::OnLoadMap(const c8* map_filename)
{
	m_LevelIndex++;
	m_pLevels[m_LevelIndex] = new CLevel();
	m_pLevels[m_LevelIndex]->Init(m_GameManager->getDevice(),m_GameManager->getFS(),m_GameManager->getDriver(),m_GameManager->getSceneMngr()/*->createNewSceneManager(false)*/);

	if (m_pLevels[m_LevelIndex]->Load(map_filename))
	{
		CreateCamera(vector3df(100,300,100));

		if (m_GameManager->m_pPC)
		{
			delete m_GameManager->m_pPC;
			m_GameManager->m_pPC = 0;
			m_GameManager->LoadPC();
		}

		m_GameManager->resetArrows();

		return true;
	}

	//Display error to console
	stringw message  = "Map: ";
	message += map_filename;
	message += " can not be loaded!";
	m_GameManager->getGUIEnvironment()->addMessageBox(L"Error Loading Map", message.c_str());

	return false;
}

/**
 * \brief Mouse and Keyboard events are handled here. 
 * Every mouse click on game object is caught, interaction with the map is handled:
 * doors lead to another map, containers display their content, NPCs start talking...
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CLevelManager::OnEvent(const SEvent& event)
{
	if(event.EventType == EET_KEY_INPUT_EVENT)
	{
		if(event.KeyInput.Key == KEY_SHIFT)
		{
			m_bShiftPressed = event.KeyInput.PressedDown;
		}
	}

	if(event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		switch(event.MouseInput.Event)
		{
			case EMIE_LMOUSE_PRESSED_DOWN:
				{
					//translate mouse click to game event:
					s32 id = m_pLevels[m_LevelIndex]->OnMouseClick();

					if ((id>-1)&&(id<(s32)m_pLevels[m_LevelIndex]->m_ListOfGameObjects.size()))
					{
						
						CGameObject* go = m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id];
						//nonexclusive list of "ifs" allows objects to be manything
						//object can be both container and pickable 
						if (m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->isContainer)
						{
							m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONCONTAINERCLICK, m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->script, id);
							return true;
						}
						if (m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->isPickable)
						{
							if(!m_GameManager->getGameGUI()->InventoryFull())
							{
								//Move item to inventory automatically
								IAnimatedMeshSceneNode* node = (IAnimatedMeshSceneNode*) m_GameManager->getSceneMngr()->getSceneNodeFromId(id);
								if(node)
								{
									stringw mesh = m_GameManager->getSceneMngr()->getMeshCache()->getMeshFilename(node->getMesh());
									stringw itemName= m_GameManager->getRootNameFromPathName(mesh);
									m_GameManager->getGameGUI()->AddPickableToInventory(itemName);
									node->remove(); //is it safe?
									//Plus do the pickup script
									m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONPICK, m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->script, id);
								}
							}
							else
							{
								m_GameManager->getGameGUI()->AddConsoleText(stringw(L"Inventory is full!"));
							}
							//return is needed here because there is a situation when action causes level change (MoveToMap)
							//and rest of the code (isNPC,isTrigger) can not be performed on new level cause it crashes naturally.
							return true;
						}
						if (m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->isTrigger)
						{
							if(m_GameManager->getGameGUI()->m_bDraggingPickableItem)
							{
								m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONUSEAGAINST, m_GameManager->getGameGUI()->m_pDraggedPickableItem->m_Script, id);
							}
							else
							{
								m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_TRIGGER_ONCLICK, m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->script, id);
							}
							//return is needed here because there is a situation when action causes level change (MoveToMap)
							//and rest of the code (isNPC check) can not be performed on new level cause it crashes naturally.
							return true;
						}
						if (m_pLevels[m_LevelIndex]->m_ListOfGameObjects[id]->isNPC)
						{
							IAnimatedMeshSceneNode* node = (IAnimatedMeshSceneNode*) m_GameManager->getSceneMngr()->getSceneNodeFromId(id);
							stringw mesh = m_GameManager->getSceneMngr()->getMeshCache()->getMeshFilename(node->getMesh());
							stringw NPCName = m_GameManager->getRootNameFromPathName(mesh);
							NPCName += ".dlg";
							m_GameManager->getGameGUI()->StartNPCDialog(NPCName);
						}
						return true;
					}
				}
				break;
			case EMIE_LMOUSE_LEFT_UP:
				break;
			case EMIE_MOUSE_MOVED:
				{
					if (!m_GameManager->getGameGUI()->m_bDraggingPickableItem)
					{
						ISceneNode* selectedGameObject = m_GameManager->getSceneMngr()->getSceneCollisionManager()->getSceneNodeFromScreenCoordinatesBB(m_GameManager->getDevice()->getCursorControl()->getPosition());
						if(selectedGameObject && selectedGameObject->getType() != ESNT_TERRAIN)
						{
							if(selectedGameObject->getID() > -1)
							{
								if(m_pLevels[m_LevelIndex]->m_ListOfGameObjects[selectedGameObject->getID()]->isPickable
									|| m_pLevels[m_LevelIndex]->m_ListOfGameObjects[selectedGameObject->getID()]->isTrigger
									|| m_pLevels[m_LevelIndex]->m_ListOfGameObjects[selectedGameObject->getID()]->isNPC
									|| m_pLevels[m_LevelIndex]->m_ListOfGameObjects[selectedGameObject->getID()]->isMonster
									|| m_pLevels[m_LevelIndex]->m_ListOfGameObjects[selectedGameObject->getID()]->isContainer)
								{
									m_bHoverOverActionItem = true;
								}
								else
								{
									m_bHoverOverActionItem = false;
								}
							}
						}
						else
						{
							m_bHoverOverActionItem = false;
						}
					}
					else
					{
						m_bHoverOverActionItem = false;
					}
				}
				break;
			case EMIE_MOUSE_WHEEL:
				{
				}
				break;
			default:
				break;
		}
	}
	return false;
}