/** 
 * \file GameManager.cpp
 * \brief CGameManager class is the boss. The manager. It creates 3D device (irrlicht of course),
 * creates GUI, and 3D environment. Calls all the initialization functions and goes to the loop.
 * 
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include <irrlicht.h>

using namespace irr;
#include "GameManager.h"

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CGameManager::CGameManager()
{
	m_ID = 0;
	m_Arrows = 0;
	m_FS = NULL;
	m_GameGUI = NULL;
	m_pLevelManager = NULL;
	m_pPC = NULL;
	m_WorkingDirectory = ".";
	m_bDoAction = false;
	m_NumberOfMaps = 0;
	m_NewGame = true;
	m_ePCMove = PC_MOVE_NO_MOVING;
}

/**
 * \brief Initialization function. Calls init of all sub classes (GUI, LevelManager, etc..)
 * \author Petar Bajic
 * \date July, 21 2008.
 */
bool CGameManager::Init()
{
	m_bInitOk = false;

	//create irrlicht device
	CreateDevice();
	//create GUI
//	m_pGuiManager = new CGameGUI();
//	m_pGuiManager->Init(this);

	//we gonna use filesystem everywhere so lets store the pointer
	m_FS = m_pDevice->getFileSystem();
	
	//Load data from XML config file
	if (!LoadDataFromXMLConfig("game_config.xml"))
		return false;

	//set window caption
	m_pDevice->setWindowCaption(m_WndCaption.c_str());

	//set working dir to whatever XML config tells you
	m_FS->changeWorkingDirectoryTo(m_WorkingDir.c_str());
	m_WorkingDirectory = m_FS->getWorkingDirectory();

	printf("%s",m_WorkingDirectory.c_str());
	//create level manager
	m_pLevelManager = new CLevelManager();

	m_GameGUI = new CGameGUI();
	if (!m_GameGUI->Init(this))
		return false;

	ClearTemp();

	//Init level after GUI creation to send GUI pointer to level manager
	//direct communicatio between level manager and gui frees game manager 
	//from constantly transfering messages back and forth between those two
	m_pLevelManager->Init(this, m_GameGUI);

	//Load First Map
	if (!m_pLevelManager->OnLoadMap(m_StartMap))
		return false;

	m_WorkingDirectory = m_FS->getWorkingDirectory();
	printf("%s",m_WorkingDirectory.c_str());

	printf("\n\nPERA\n\n");

	//Load Player Character
	m_pPC = new CPlayerCharacter(this);
	stringc playerFile = stringc("game/") + m_PlayerConfigFile;
	if (!m_pPC->Load(m_pDevice,m_pSceneManager,playerFile.c_str()))
		return false;

	m_WorkingDirectory = m_FS->getWorkingDirectory();
	printf("%s",m_WorkingDirectory.c_str());

	m_pPC->setPosition(m_pLevelManager->GetStartPosition());

	m_pScriptEngine = new CScript();
	if(!m_pScriptEngine->Init(this, m_pDevice,"media/Scripts/actions.script"))
		return false;

	m_pDevice->getCursorControl()->setVisible(false);
	m_MousePointerTexture = m_pDriver->getTexture("media/Icons/mouse_pointer.png");
	m_MousePointerActionTexture = m_pDriver->getTexture("media/Icons/mouse_action.png");
	m_MousePointerPickableDraggingTexture = m_pDriver->getTexture("media/Icons/mouse_pick.png");
	if((m_MousePointerTexture == 0)||(m_MousePointerActionTexture == 0)||(m_MousePointerPickableDraggingTexture == 0))
		return false;

	m_Arrows = new CGoToArrows(m_pDevice,m_pSceneManager,m_pSceneManager->getRootSceneNode(),-1,SColor(255,150,0,150),0);

	//colision response
	m_pPC->addAnimator(m_pLevelManager->GetObstacleMetaTriangleSelector());

	InitNPCDialogs();

	m_bInitOk = true;

	stringw message  = "Welcome to game example!\n\nThis game is simple example of what can be made with Level Editor. At the moment this is kind of 3rd person point and click adventure. \n\nClick around to move and interact with objects. Adjust camera with right mouse click (press hold and move). \n";
	m_GameGUI->AddMsgBox(L"Welcome!", message.c_str());

	return true;
}

//Parse all maps and find all NPCs and load all their dialogs... OR
//Load dialogs found in media/Dialogs dir ... OR
//Have user define all dialogs in game config?
//Using first option now.
void CGameManager::InitNPCDialogs()
{
	for(u32 u=0; u<m_NumberOfMaps; u++)
	{
		//Find NPCs on the map
		stringc NPCNames[MAX_NUMBER_OF_NPCS_PER_MAP];
		u32 numberOfNPC = m_pLevelManager->FindNPCsOnMap(stringc("maps/") + m_Maps[u], NPCNames);
		for(u32 i=0; i<numberOfNPC; i++)
		{
			m_GameGUI->AddGameDialog(NPCNames[i] + stringc(".dlg"));
		}
	}
}

void CGameManager::PCMoveEnd()
{
	m_bDoAction = true;
	m_ePCMove = PC_MOVE_STOPS_MOVING;
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CGameManager::~CGameManager()
{
}

bool CGameManager::LoadPC(stringc playerFile)
{
	//Load Player Character
	m_pPC = new CPlayerCharacter(this);
	if (!m_pPC->Load(m_pDevice,m_pSceneManager,playerFile.c_str()))
		return false;

	return true;
}

void CGameManager::ClearTemp()
{
	//getting all files in dir is platform specific
	//for windows one would use FindFirstFile function and so on...
	//to avoid this platform dependency, we will search for our maps
	//loaded from xml, and ignore other files that end up in temp
	u32 index = 0;
	for(index = 0; index<m_NumberOfMaps; index++)
	{
		stringc mapPath = stringc("save/temp/") + m_Maps[index];
		remove(mapPath.c_str());
	}
}

void CGameManager::ClearSaved()
{
	//getting all files in dir is platform specific
	//for windows one would use FindFirstFile function and so on...
	//to avoid this platform dependency, we will search for our maps
	//loaded from xml, and ignore other files that end up in temp
	u32 index = 0;
	for(index = 0; index<m_NumberOfMaps; index++)
	{
		stringc mapPath = stringc("save/") + m_Maps[index];
		remove(mapPath.c_str());
	}
}

stringc CGameManager::getSavedMap(stringc mapname)
{
	stringc mapPath = stringc("save/temp/") + mapname;

	//find if there is saved map
	if(m_FS->existFile(mapPath.c_str()))
	{
		return mapPath;
	}
	
	mapPath = stringc("save/") + mapname;
	
	if (m_FS->existFile(mapPath.c_str()))
	{
		return mapPath;
	}

	mapPath = stringc("maps/") + mapname;

	return mapPath;
}

stringw CGameManager::getCurrentMapName()
{
	return m_pLevelManager->getCurrentMapName();
}

//Second parameter is used only when there is more then one entry to the map, to specify where the player will show up.
void CGameManager::ChangeLevel(stringc mapname, s32 startPositionID)
{
	//save current level to temp;
	m_pLevelManager->SavePreviousLevelToTemp();
	//m_GameGUI->SaveNPCDialogsToTemp();

	//TODO: make sure the new level is different then current one.
	//rephrase: make sure we are not loading the level we are currently in.

	stringc mapPath = getSavedMap(mapname);
	m_pLevelManager->OnLoadMap(mapPath);

	m_pPC->ReinitModel(m_pSceneManager);
	if(startPositionID > 0)
	{
		vector3df startPos = m_pLevelManager->GetObjectPosition(startPositionID);
		m_pPC->setPosition(startPos);
		m_pLevelManager->SetCameraPos(startPos);
	}
	else
	{
		m_pPC->setPosition(m_pLevelManager->GetStartPosition());
	}
	m_pPC->removeAnimators();
	m_pPC->addAnimator(m_pLevelManager->GetObstacleMetaTriangleSelector());

	resetArrows();
}

void CGameManager::Restart()
{
	//load latest saved level
	LoadGame();
}

void CGameManager::ClearInventory()
{
	//clear inventory
	m_GameGUI->ClearInventory();
}

void CGameManager::LoadGame()
{
	m_NewGame = false;
	//clear temp
	ClearTemp();
	//Clear Inventory
	ClearInventory();
	//get latest level and load
	stringc playerFile = stringc("save/") + m_PlayerConfigFile;
	if(m_FS->existFile(playerFile.c_str())) {
		//m_LoadedMapName is loaded with PC
		LoadPC(playerFile);
		//Load Level
		m_pLevelManager->OnLoadMap(stringc("save/") + m_LoadedMapName);
		//Adjst player to new Level
		m_pPC->ReinitModel(m_pSceneManager);
		m_pPC->SetLoadedPosition();
		m_pPC->removeAnimators();
		m_pPC->addAnimator(m_pLevelManager->GetObstacleMetaTriangleSelector());
		//Move Camera above PC
		m_pLevelManager->MoveCamera(m_pPC->getPosition());
		//Reset arrows
		resetArrows();
		//Load Dialogs
		m_GameGUI->LoadNPCDialogs();
	}
	else
	{
		//restart the game
		//Display error to console
		stringw message  = "There is no saved game!";
		m_GameGUI->AddMsgBox(L"Error Loading Game", message.c_str());
	}
}

void CGameManager::SaveGame()
{
	if(m_NewGame)
	{
		//Playing from start, want to save? remove existing saved levels...
		ClearSaved();
	}

	m_NewGame = false;

	//Save Player Character Stats
	m_pPC->Save();

	//Save Scene
	m_pLevelManager->SaveLevels();

	//Save NPC conversation
	m_GameGUI->SaveNPCDialogs();
}

void CGameManager::resetArrows()
{
	if(m_Arrows)
	{
		delete m_Arrows;
		m_Arrows = 0;
		m_Arrows = new CGoToArrows(m_pDevice,m_pSceneManager,m_pSceneManager->getRootSceneNode(),-1,SColor(255,150,0,150),0);
	}
}

void CGameManager::HandleDeath()
{
	m_pPC->Dies();
	m_GameGUI->DisplayDeathWindow(m_pGUIEnvironment);
}

/**
 * \brief Game drawing loop. Draws GUI (2D) and current level (3D).
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameManager::Update(f32 elapsed_time)
{
	if(m_bDoAction)
	{
		m_pLevelManager->Action(m_ClickedNodeID);
		m_bDoAction = false;
		return;
	}

	m_pDriver->beginScene(true, true, SColor(255,100,101,140));
	if (m_pPC && m_pPC->isAlive())
	{
		m_pPC->update(elapsed_time);
		if(m_ePCMove == PC_MOVE_STARTS_MOVING)
		{
			m_pLevelManager->MoveCamera(m_pPC->getPosition());
		}

		if(m_pPC->getPosition().Y < -50)
		{
			//you fell off the world?
			HandleDeath();
		}
	}
	m_pSceneManager->drawAll();
	m_pLevelManager->OnRender();
	m_pGUIEnvironment->drawAll();
	
	IGUIFont* font = m_pGUIEnvironment->getBuiltInFont();
	m_GameGUI->healthBar->renderGUIBars(m_pDriver, font);

	if(m_GameGUI->m_wnd_charSheet)
	{
		//Im sorry for this code TODO: make guiBars gui elements!
		int x = m_GameGUI->m_wnd_charSheet->getAbsolutePosition().UpperLeftCorner.X;
		int y = m_GameGUI->m_wnd_charSheet->getAbsolutePosition().UpperLeftCorner.Y;
		for(u32 i=0; i<m_pPC->m_ListOfAbilities.size(); i++)
		{
			if(m_GameGUI->cs_ability_bars[i])
			{
				m_GameGUI->cs_ability_bars[i]->setPos(x + 20, y+60+i*30, x + 120, y+80+i*30);
				m_GameGUI->cs_ability_bars[i]->renderGUIBars(m_pDriver, font);
			}
		}
		for(u32 i=0; i<m_pPC->m_ListOfSkills.size(); i++)
		{
			if(m_GameGUI->cs_skill_bars[i])
			{
				m_GameGUI->cs_skill_bars[i]->setPos(x + 20, y+270+i*30, x + 120, y+290+i*30);
				m_GameGUI->cs_skill_bars[i]->renderGUIBars(m_pDriver, font);
			}
		}
		/*m_GameGUI->cs_health->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_str->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_exp->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_dex->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_int->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_attack->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_defence->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_magic->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_gardenning->renderGUIBars(m_pDriver, font);
		m_GameGUI->cs_level->renderGUIBars(m_pDriver, font);*/
	}

	if(m_GameGUI->m_bDraggingPickableItem)
	{
		m_pGUIEnvironment->getVideoDriver()->draw2DImage(m_MousePointerPickableDraggingTexture,
			m_pDevice->getCursorControl()->getPosition()+position2di(-15, -15),
				core::rect<s32>(0,0,82,78), 0, 
				video::SColor(255,255,255,255), true);
		m_pGUIEnvironment->getVideoDriver()->draw2DImage(m_GameGUI->m_pDraggedPickableItem->m_IconTexture,m_pDevice->getCursorControl()->getPosition()+position2di(-25, -25));
	}
	else if (m_pLevelManager->m_bHoverOverActionItem)
	{
		m_pGUIEnvironment->getVideoDriver()->draw2DImage(m_MousePointerActionTexture,
			m_pDevice->getCursorControl()->getPosition(),
				core::rect<s32>(0,0,50,50), 0, 
				video::SColor(255,255,255,255), true);
	}
	else
	{
		m_pGUIEnvironment->getVideoDriver()->draw2DImage(m_MousePointerTexture,
			m_pDevice->getCursorControl()->getPosition(),
				core::rect<s32>(0,0,50,50), 0, 
				video::SColor(255,255,255,255), true);
	}

	m_pDriver->endScene();
}

void CGameManager::DisplayError()
{
	m_pDriver->beginScene(true, true, SColor(255,100,101,140));
	m_pSceneManager->drawAll();
	m_pGUIEnvironment->drawAll();
	m_pDriver->endScene();
}

stringw CGameManager::getRootNameFromPathName(stringw meshName)
{
	stringw rootName = meshName.subString(meshName.findLast('/')+1,meshName.size());
	rootName = rootName.subString(0,rootName.findLast('.'));
	return rootName;
}

stringc CGameManager::getRootNameFromPathName(stringc meshName)
{
	stringc rootName = meshName.subString(meshName.findLast('/')+1,meshName.size());
	rootName = rootName.subString(0,rootName.findLast('.'));
	return rootName;
}

/**
 * \brief Creates the Irrlicht device and get pointers to the main subsytems
 * for later use, the Game manager is the central interface point to the rendering engine
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameManager::CreateDevice()
{
	//create irrlicht device and send "this" as event receiver meaning this class should 
	//implement OnEvent function and handle its own events...
	m_pDevice = createDevice( video::EDT_DIRECT3D9, dimension2d<u32>(900, 650), 16, false, false, false, this);
	//m_pDevice->setResizeAble(true);
   	m_pDriver = m_pDevice->getVideoDriver();
    m_pSceneManager = m_pDevice->getSceneManager();
	m_pGUIEnvironment = m_pDevice->getGUIEnvironment();
	//change to m_WorkingDirectory
	IGUIFont* font = m_pGUIEnvironment->getFont("../media/Garamond14.xml");
	m_pGUIEnvironment->getSkin()->setFont(font);

}

/**
 * \brief Loads config data from given XML file
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CGameManager::LoadDataFromXMLConfig(stringc filename)
{
	io::IXMLReader* xml = m_FS->createXMLReader(filename.c_str());

	if (!xml)
	{
		//Display error message and exit

		return false;
	}

	while(xml->read())
	{
		switch(xml->getNodeType())
		{
		case io::EXN_ELEMENT:
			{
				stringw figo;
				if (core::stringw("MapStart") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"filename");
					m_StartMap = figo.c_str();
				}
				else if (core::stringw("PlayerConfig") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"filename");
					m_PlayerConfigFile = figo.c_str();
				}
				else if (core::stringw("WorkingDir") == xml->getNodeName())
				{
					//New Folder, create node and step in.
					figo = xml->getAttributeValue(L"filename");
					m_WorkingDir = figo.c_str();
				}
				else if (core::stringw("WindowCaption") == xml->getNodeName())
				{
					//New Folder, create node and step in.
					figo = xml->getAttributeValue(L"text");
					m_WndCaption = figo.c_str();
				}
				else if (stringw("Level") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"filename");
					m_Maps[m_NumberOfMaps] = figo;
					m_NumberOfMaps++;
				}
			}
			break;
		}
	}

	xml->drop(); // don't forget to delete the xml reader

	return true;
}

/**
 * \brief Returns a pointer to the Irrlicht Device subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
IrrlichtDevice* CGameManager::getDevice()
{
	return m_pDevice;
}

/**
 * \brief Returns a pointer to the Irrlicht Driver subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
IVideoDriver* CGameManager::getDriver()
{
	return m_pDriver;
}

/**
 * \brief Changes working directory to default one.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CGameManager::backToWorkingDirectory()
{
	m_FS->changeWorkingDirectoryTo(m_WorkingDirectory.c_str());
}

/**
 * \brief Returns a pointer to the Irrlicht SceneManager subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
ISceneManager* CGameManager::getSceneMngr()
{
	return m_pSceneManager;
}

/**
 * \brief Returns a pointer to the Irrlicht GUI subsystem
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
IGUIEnvironment* CGameManager::getGUIEnvironment()
{
	return m_pGUIEnvironment;
}

/**
 * \brief Displays GUI Window with content of container object with given id
 * \author Petar Bajic 
 * \date July, 21 2008.
 
void CGameManager::DisplayContainerContent(s32 id)
{
	m_GameGUI->DisplayContainerContent(id, m_pDriver, m_pGUIEnvironment, m_pLevelManager);
}*/

/**
 * \brief Main event handler derived from IEventHandler, this will be passed down 
 * to the current states keyboard handler. The state controls its own keyboard events
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CGameManager::OnEvent(const SEvent& event)
{
	if (!m_pDriver || !m_bInitOk)
		return false;

	//Gui Handles its own events here
	if(m_GameGUI->OnEvent(event))
		return false; //false will let irrlicht handle events too

	//3D level environment handles events here
	if (m_pLevelManager->OnEvent(event))
		return false; //false will let irrlicht handle events too

	if (event.EventType == EET_KEY_INPUT_EVENT)
	{
		if(event.KeyInput.PressedDown){
			if(event.KeyInput.Key==KEY_F2){
				//if(console->isVisible())
				return true;
			}
			else{
				//m_bKeys[event.KeyInput.Key] = event.KeyInput.PressedDown;
				// Pass input down to the specific game state keyboard handler
			}
		}
	}

	if (event.EventType == EET_MOUSE_INPUT_EVENT)
	{
		// Pass input down to the specific game state mouse handler
		switch(event.MouseInput.Event)
		{
			case EMIE_LMOUSE_PRESSED_DOWN:
				{
					//Player move event - this could be handled in PC class OnEvent function (add one if need rises)
					const ISceneNode* hitNode = 0;
					vector3df instersection_point; //intersection of terrain and mouse click
					triangle3df instersection_triangle; //not used, but needed as parameter for function call
					line3d<f32> picking_line = getSceneMngr()->getSceneCollisionManager()->getRayFromScreenCoordinates(getDevice()->getCursorControl()->getPosition());
					IMetaTriangleSelector* selector = m_pLevelManager->GetLevelMetaTriangleSelector();
					getSceneMngr()->getSceneCollisionManager()->getCollisionPoint(picking_line,selector,instersection_point,instersection_triangle,hitNode);

					if ((hitNode != 0)&&(hitNode->getID() >= 0)) 
					{
						//hitNode = getSceneMngr()->getSceneNodeFromId(chitNode->getID());
						m_ClickedNodeID = hitNode->getID();
						if(m_pLevelManager->isNodeTerrain(hitNode))
						{
							if(m_GameGUI->m_bDraggingPickableItem)
							{
								//drop pickable item on terrain
								m_GameGUI->m_bDraggingPickableItem = false;
								m_pLevelManager->DropPickableToMap(m_GameGUI->m_pDraggedPickableItem,instersection_point + vector3df(0,1,0));
								m_GameGUI->m_pDraggedPickableItem = 0;
							}
							else
							{
								//move pc to point where clicked on the terrain.
								m_pPC->setTargetPosition(instersection_point); 
								m_Arrows->Play(instersection_point + vector3df(0,5,0));
								m_ePCMove = PC_MOVE_STARTS_MOVING;
							}
						}
						else if(m_pLevelManager->isNodeActionObject(hitNode))
						{
							if(_isNodeClose(m_pPC->node->getPosition(),hitNode->getPosition()))
							{
								//skip moving, interact right away
								m_bDoAction = true;
							}
							else
							{
								//move pc to approach hit node as action object
								m_pPC->setTargetNode(hitNode);
								m_ePCMove = PC_MOVE_STARTS_MOVING;
							}
						}
						else 
						{
							//click on the non terrain non action item
							m_GameGUI->AddConsoleText(L"Nothing to be had there...");
						}
					}
				}
				break;
		}
	}

	return false;
}

bool CGameManager::_isNodeClose(vector3df pos1, vector3df pos2)
{
	if(pos1.getDistanceFrom(pos2) < NEAR_OBJECT_DISTANCE)
	{
		return true;
	}
	return false;
}


