/** 
 * \file GameObject.cpp
 * \brief GameObject class is used for storing Game object attributes and data.
 * CGameObject class handles mesh name, id, and parameters like isContainer, isNPC, etc...
 * it also handles list of pickable items (if the object is container) of the object.
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "GameObject.h"
#include "../GameManager.h"

CGameObject::CGameObject()
{
	isContainer = false;
	isPickable = false;
	isTrigger = false;
	isNPC = false;
	isMonster = false;
	m_IconTexture = 0;
}

/**
 * Advanced constructor.
 */
CGameObject::CGameObject(IXMLReader* xml, IVideoDriver* driver)
{
	animations.clear();
	isAnimated = false;
	isContainer = false;
	isMonster = false;
	isAnchored = false;
	isNPC = false;
	isPickable = false;
	isTrigger = false;
	isInvisible = false;
	isStatic = false;
	state = "";
	description = L"No description specified";
	m_Driver = driver;

	if(xml)
	{
		LoadPropertiesFromXMLFile(xml);
		xml->drop();
	}
}

/**
 * Advanced constructor. Used for pickable items placed in container objects loaded from map files.
 */
CGameObject::CGameObject(stringw _root, s32 _id, IXMLReader* xml, IVideoDriver* driver)
{
	s32 position = _root.findLastChar(L"/",1);
	stringc _name = _root.subString(position+1,_root.size()-position);
	stringc _path = _root.subString(0,position+1);

	animations.clear();
	isAnimated = false;
	name = _name;
	mesh = _path;
	root = _root;
	id = _id;
	isContainer = false;
	isMonster = false;
	isAnchored = false;
	isNPC = false;
	isPickable = false;
	isTrigger = false;
	isInvisible = false;
	isStatic = false;
	isTerrain = false;
	m_IconTexture = NULL;
	m_Driver = driver;
	description = L"No description specified";
	script = _name + ".script"; //default, but can be different
	icon = _name + ".png"; //default, but can be different

	pos = vector3df(0,0,0);
	rot = vector3df(0,0,0);
	scale = vector3df(1,1,1);

	if(xml)
	{
		LoadPropertiesFromXMLFile(xml);
		xml->drop();
	}
}

CGameObject::~CGameObject()
{
}

stringw CGameObject::GetPickableItemRoot(s32 id)
{
	return m_ListOfPickableItems[id]->root;
}

s32 CGameObject::GetPickableItemID(s32 id)
{
	return m_ListOfPickableItems[id]->id;
}

void CGameObject::LoadPropertiesFromXMLFile(IXMLReader* xml)
{
	while(xml->read())
	{
		switch(xml->getNodeType())
		{
		case io::EXN_ELEMENT:
			{
				stringw figo;
				if (stringw("Name") == xml->getNodeName())
				{
					name = xml->getAttributeValue(L"value");
				}
				else if (stringw("Model") == xml->getNodeName())
				{
					//Why += ? because path is already stored in mesh, so we just add model filename
					mesh += xml->getAttributeValue(L"value");
				}
				else if (stringw("Icon") == xml->getNodeName())
				{
					icon = xml->getAttributeValue(L"value");
				}
				else if (stringw("Description") == xml->getNodeName())
				{
					description = xml->getAttributeValue(L"value");
				}
				else if (stringw("Script") == xml->getNodeName())
				{
					script = xml->getAttributeValue(L"value");
				}
				else if (stringw("Pickable") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"value");
					if (figo == L"true")
					{
						isPickable = true;
					}
					else
					{
						isPickable = false;
					}
				}
				else if (stringw("Container") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"value");
					if (figo == L"true")
					{
						isContainer = true;
					}
					else
					{
						isContainer = false;
					}
				}
				else if (stringw("Monster") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"value");
					if (figo == L"true")
					{
						isMonster = true;
					}
					else
					{
						isMonster = false;
					}
				}
				else if (stringw("NPC") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"value");
					if (figo == L"true")
					{
						isNPC = true;
					}
					else
					{
						isNPC = false;
					}
				}
				else if (stringw("Trigger") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"value");
					if (figo == L"true")
					{
						isTrigger = true;
					}
					else
					{
						isTrigger = false;
					}
				}
				else if (stringw("Animation") == xml->getNodeName())
				{
					TAnimationRange anim;
					anim.startFrame = xml->getAttributeValueAsInt(L"start");
					anim.endFrame = xml->getAttributeValueAsInt(L"end");
					anim.animName = xml->getAttributeValue(L"name");
					animations.push_back(anim);
					isAnimated = true;
				}
				else if (stringw("Invisible") == xml->getNodeName())
				{
					figo = xml->getAttributeValue(L"value");
					if (figo == L"true")
					{
						isInvisible = true;
					}
					else
					{
						isInvisible = false;
					}
				}
			}
		}
	}

	if(isPickable) m_IconTexture = m_Driver->getTexture(stringw(ICON_DIR) + icon);
}
/*
void CGameObject::AddPickableItem(stringc _root, s32 _id, IXMLReader* xml)
{
	s32 position = _root.findLastChar("/",1);
	stringc name = _root.subString(position+1,_root.size()-position);
	stringc path = _root.subString(0,position+1);
	CGameObject* pickObject = new CGameObject(path, name, false);
	pickObject->root = _root;
	pickObject->id = _id;

	if(xml)
	{
		pickObject->LoadPropertiesFromXMLFile(xml);
		xml->drop();
	}

	m_ListOfPickableItems.push_back(pickObject);
}*/

void CGameObject::AddPickableItem(CGameObject* pick)
{
	m_ListOfPickableItems.push_back(pick);
}

CGameObject* CGameObject::GetPickableItem(s32 id)
{
	return m_ListOfPickableItems[id];
}

void CGameObject::ClearPickableItems()
{
	m_ListOfPickableItems.clear();
}

void CGameObject::ClearScriptActions()
{
	m_ListOfScriptActions.clear();
}

s32 CGameObject::GetNumberOfPickableItems()
{
	return m_ListOfPickableItems.size();
}