/** 
 * \file Level.cpp
 * \brief File Level.cpp takes care of level structure - it hold together all game objects on the level.
 *
 * takes care of saving/loading level
 * manages game object arrays (adding deleting game objects)
 * translates events to game objects -> on mouse click, on rotate etc..
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "Level.h"
	

CLevel::CLevel()
{
	m_ID = 0;
	m_startPos = vector3df(0,0,0);
}

CLevel::~CLevel()
{
}

bool CLevel::Init(IrrlichtDevice* device, IFileSystem* fs, IVideoDriver* driver, ISceneManager* smgr)
{
	m_ID = 0;
	m_pFS = fs;
	m_pDriver = driver;
	m_SMGR = smgr;
	m_pDevice = device;
	return true;
}

/**
 * \brief Returns player start position if flagged on the map.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
vector3df CLevel::GetStartPosition()
{
	return m_startPos;
}

stringw CLevel::GetRootFromPath(stringw path)
{
	s32 position = path.findLastChar(L"/",1);
	stringc filename = path.subString(position+1,path.size()-position);
	position = filename.findLastChar(".",1);
	stringw _root = filename.subString(0,position);
	return _root;
}

void CLevel::Save(stringc map)
{
	stringc workingDirectory = m_pDevice->getFileSystem()->getWorkingDirectory();
	printf("%s",workingDirectory.c_str());

	IXMLWriter* writer = m_pFS->createXMLWriter(map.c_str());
	writer->writeXMLHeader();
	WriteSceneNode(writer, m_SMGR->getRootSceneNode());
	writer->drop();
}

/**
 * \brief Writes a scene node attributes to xml file.
 *
 * And calls itself recursivly.
 * Optimizations can be made to reduce file size:
 * Default attributes (eg. Scale: 1,1,1) can be skipped in saving process, and loaded only if
 * different from default (1,1,1).
 *
 * \author Petar Bajic 
 * \date January, 21 2010.
 */
void CLevel::WriteSceneNode(IXMLWriter* writer, ISceneNode* node)
{
	if (!writer || !node)
		return;

	if (node == m_SMGR->getRootSceneNode())
	{
		writer->writeElement(L"LevelEditorMap", false);
		writer->writeLineBreak();
		writer->writeLineBreak();
	}
	else if (node->getID() > -1)
	{
		CGameObject* gameObject = getGameObjectFromID(node->getID());

		if(gameObject)
		{
			writer->writeElement(L"GameObject", false);
			writer->writeLineBreak();

			// write properties
			io::IAttributes* attr = m_pFS->createEmptyAttributes(m_pDriver);
			attr->addString("Name",node->getName());
			attr->addInt("ID",node->getID());
			attr->addString("Mesh",gameObject->mesh.c_str());
			attr->addVector3d("Position",node->getPosition());
			attr->addVector3d("Rotation",node->getRotation());
			attr->addVector3d("Scale",node->getScale());
			if (gameObject->isInvisible)
				//adding attribute isInvisible
				attr->addBool("isInvisible",true);
			if (gameObject->isTrigger)
				//adding atribute isTrigger
				attr->addBool("isTrigger",true);
			if (gameObject->isStatic)
				//adding atribute isStatic
				attr->addBool("isStatic",true);
			if (gameObject->isTerrain)
				//adding atribute isTerrain
				attr->addBool("isTerrain",true);
			if (gameObject->isPickable)
				//adding atribute isPickable
				attr->addBool("isPickable",true);
			if (gameObject->isMonster)
				//adding atribute isMonster
				attr->addBool("isMonster",true);
			if (gameObject->isNPC)
				//adding atribute isNPC
				attr->addBool("isNPC",true);
			if (gameObject->isContainer)
				//adding attribute isContainer
				attr->addBool("isContainer",true);
			if (gameObject->script != stringw(L""))
				attr->addString("Script",gameObject->script.c_str());
			if (gameObject->state != stringw(L""))
				attr->addString("State",gameObject->state.c_str());
			attr->write(writer);
			//writer->writeLineBreak();
			attr->drop();

			//Write container content
			if (gameObject->isContainer)
			{
				writer->writeElement(L"Container",false); 
				writer->writeLineBreak();
				for ( s32 index = 0; index < gameObject->GetNumberOfPickableItems(); index++)
				{
					writer->writeElement(L"Pickables",true,L"root",gameObject->GetPickableItemRoot(index).c_str(), L"id", stringw(gameObject->GetPickableItemID(index)).c_str());
					writer->writeLineBreak();
				}
				writer->writeClosingTag(L"Container"); 
				writer->writeLineBreak();
			}
		}
	}

	// write children

	core::list<ISceneNode*>::ConstIterator it = node->getChildren().begin();
	for (; it != node->getChildren().end(); ++it)
		WriteSceneNode(writer, (*it));

	if ((node == m_SMGR->getRootSceneNode())||(node->getID() > -1))
	{
		if (node == m_SMGR->getRootSceneNode())
		{
			writer->writeClosingTag(L"LevelEditorMap");
			writer->writeLineBreak();
		}
		else
		{
			CGameObject* tempGameObject = getGameObjectFromID(node->getID());
			if(tempGameObject)
			{
				writer->writeClosingTag(L"GameObject");
				writer->writeLineBreak();
				writer->writeLineBreak();
			}
		}
	}
}


/**
 * \brief Reads a scene nodes from xml file.
 *
 * \author Petar Bajic 
 * \date May, 21 2009.
 */
void CLevel::ReadSceneNode(IXMLReader* reader)
{
	CGameObject* gameObject;
	stringw rootName;

	//loop through scene nodes in xml
	while(reader->read())
	{
		switch(reader->getNodeType())
		{
		case io::EXN_ELEMENT:
			{
				if (stringw("attributes") == reader->getNodeName())
				{
					//load node attributes
					IAttributes* attr = m_pFS->createEmptyAttributes(m_pDriver);
					attr->read(reader);
					stringc meshPath = attr->getAttributeAsString("Mesh");
					IAnimatedMesh* m = m_SMGR->getMesh(meshPath.c_str());
					if (m)
					{
						//adding new game object to the map
						s32 position = meshPath.findLastChar(".",1);
						stringw _root = meshPath.subString(0,position);
						stringw xmlProperties = _root + L".xml";
						IXMLReader* xml = m_pFS->createXMLReader(stringc(xmlProperties.c_str()).c_str());
						//Create GO
						gameObject = new CGameObject(xml,m_pDriver);
						gameObject->root = _root;
						ISceneNode* node = m_SMGR->addAnimatedMeshSceneNode(m);
						node->setMaterialFlag(EMF_LIGHTING, false);
						((IAnimatedMeshSceneNode*)node)->setAnimationSpeed(10);

						// !!?
						ITriangleSelector *selector = m_SMGR->createTriangleSelector(((IAnimatedMeshSceneNode*)node)->getMesh(), node);
						node->setTriangleSelector(selector);
						

						node->setName(attr->getAttributeAsString("Name"));
						node->setID(attr->getAttributeAsInt("ID"));
						node->setPosition(attr->getAttributeAsVector3d("Position"));
						node->setRotation(attr->getAttributeAsVector3d("Rotation"));
						node->setScale(attr->getAttributeAsVector3d("Scale"));
						node->setVisible(!attr->getAttributeAsBool("isInvisible"));
						
						gameObject->name = node->getName();
						gameObject->id = node->getID();
						gameObject->mesh = meshPath;
						gameObject->pos = node->getPosition();
						gameObject->rot = node->getRotation();
						gameObject->scale = node->getScale();
						gameObject->script = attr->getAttributeAsStringW("Script");
						gameObject->state = attr->getAttributeAsStringW("State");

						if(stringc(node->getName()) == "Start Flag")
						{
							//record start flag position
							m_startPos = node->getPosition();
							m_startPos.Y += 10; //Above the ground, so that it falls down easily like a feather...
						}

						gameObject->isContainer = attr->getAttributeAsBool("isContainer");
						gameObject->isAnchored = attr->getAttributeAsBool("isAnchored");
						gameObject->isNPC = attr->getAttributeAsBool("isNPC");
						gameObject->isMonster = attr->getAttributeAsBool("isMonster");
						gameObject->isPickable = attr->getAttributeAsBool("isPickable");
						gameObject->isTrigger = attr->getAttributeAsBool("isTrigger");
						gameObject->isStatic = attr->getAttributeAsBool("isStatic");
						gameObject->isTerrain = attr->getAttributeAsBool("isTerrain");
						gameObject->isInvisible = attr->getAttributeAsBool("isInvisible");

						if(!gameObject->isInvisible/* && !gameObject->isPickable*/)
						{
							m_LevelMetaTriangleSelector->addTriangleSelector(node->getTriangleSelector());
							if(!gameObject->isPickable && !gameObject->isMonster && !gameObject->isNPC)
							{
								m_ObstacleMetaTriangleSelector->addTriangleSelector(node->getTriangleSelector());
							}
						}

						if((gameObject->isTrigger)&&(gameObject->state.size() == 0))
						{
							//load default state if state is not loaded from map file
							stringc temp = m_pDevice->getFileSystem()->getWorkingDirectory();
							m_pDevice->getFileSystem()->changeWorkingDirectoryTo("media/Scripts/Static");
							io::IXMLReader* xml = m_pDevice->getFileSystem()->createXMLReader(gameObject->script.c_str());
							while(xml && xml->read())
							{
								switch(xml->getNodeType())
								{
								case io::EXN_ELEMENT:
									{
										if (stringw("State") == xml->getNodeName())
										{
											gameObject->state = xml->getAttributeValue(L"value");
										}
									}
									break;
								}
							}
							if (xml)
								xml->drop(); // don't forget to delete the xml reader
							m_pDevice->getFileSystem()->changeWorkingDirectoryTo(temp.c_str());
						}

						if(gameObject->isPickable)
						{
							rootName = GetRootFromPath(meshPath);
							stringw icon = rootName + ".png";
							gameObject->m_IconTexture = m_pDriver->getTexture(stringw(L"media/icons/") + icon);
						}

						if(gameObject->isAnimated)
						{
							//set model to idle animation
							((IAnimatedMeshSceneNode*)node)->setFrameLoop(gameObject->getAnimStart("Idle"),gameObject->getAnimEnd("Idle"));
						}

						m_ListOfGameObjects.push_back(gameObject);
					}
				}

				//Load container content from xml file to GameObjects list of pickable items
				if (stringw("Pickables") == reader->getNodeName())
				{
					//Add pickable game object (PGO) to container game object (CGO)
					stringc pickRoot = reader->getAttributeValue(L"root");
					stringc idStr = reader->getAttributeValue(L"id");
					s32 id = atoi(idStr.c_str());
					stringw xmlProperties = pickRoot + L".xml";
					IXMLReader* xml = m_pFS->createXMLReader(stringc(xmlProperties.c_str()).c_str());
					//Create pickable GO and add it to container GO
					CGameObject* pickGO = new CGameObject(pickRoot,id,xml,m_pDriver);
					pickGO->isPickable = true;
					gameObject->AddPickableItem(pickGO);
				}
			}
			break;
		}
	}
}

bool CLevel::Load(stringc map_filename)
{
	m_ID = 0; //cleaning old map
	m_SMGR->clear();
	m_ListOfGameObjects.clear();
	m_MapName = map_filename.subString(map_filename.findLast('/')+1,map_filename.size());

	stringc workingDirectory = m_pDevice->getFileSystem()->getWorkingDirectory();
	printf("%s",workingDirectory.c_str());

	m_LevelMetaTriangleSelector = m_SMGR->createMetaTriangleSelector();
	m_ObstacleMetaTriangleSelector = m_SMGR->createMetaTriangleSelector();
	IXMLReader* reader = m_pFS->createXMLReader(map_filename.c_str());
	if(reader)
	{
		ReadSceneNode(reader);
		reader->drop();
		return true;
	}

	return false;
}

bool CLevel::EraseElement(int id)
{
	list<CGameObject*>::Iterator it = m_ListOfGameObjects.begin();
	
	for (; it != m_ListOfGameObjects.end(); ++it)
	{
		if((*it)->id == id)
		{
			m_ListOfGameObjects.erase(it);
			return true;
		}
	}

	return false;
}

void CLevel::AddObjectToScene(CGameObject* pick, vector3df position)
{
	CGameObject* go = getGameObjectFromID(pick->id);
	if(go)
	{
		//error! There already exists go with this ID! Unexpected error! Abort! Abort!
		return;
	}
	m_ListOfGameObjects.push_back(pick);
	
	//add to scene
	IAnimatedMesh* m = m_SMGR->getMesh(pick->mesh.c_str());
	if (m)
	{
		//adding new game object to the map
		ISceneNode* node = m_SMGR->addAnimatedMeshSceneNode(m);
		node->setMaterialFlag(EMF_LIGHTING, false);
		((IAnimatedMeshSceneNode*)node)->setAnimationSpeed(10);

		// !!?
		ITriangleSelector *selector = m_SMGR->createTriangleSelector(((IAnimatedMeshSceneNode*)node)->getMesh(), node);
		node->setTriangleSelector(selector);
		m_LevelMetaTriangleSelector->addTriangleSelector(selector);

		node->setName(pick->name);
		node->setID(pick->id);
		node->setPosition(position);
		node->setRotation(pick->rot);
		node->setScale(pick->scale);

		pick->pos = position;
	}
}

CGameObject* CLevel::getGameObjectFromID(int id)
{
	list<CGameObject*>::Iterator it = m_ListOfGameObjects.begin();
	
	for (; it != m_ListOfGameObjects.end(); ++it)
	{
		if((*it)->id == id)
		{
			return *it;
		}
	}

	return 0;
}

void  CLevel::RemoveContainerContent(int containerID)
{
	//m_ListOfGameObjects[containerID]->ClearPickableItems();
	getGameObjectFromID(containerID)->ClearPickableItems();
}

void CLevel::AddContainerItem(int containerID, CGameObject* pick)
{
	//m_ListOfGameObjects[containerID]->AddPickableItem(itemName.c_str());
	getGameObjectFromID(containerID)->AddPickableItem(pick);
}

int CLevel::GetContainerNumberOfItems(int containerID)
{
	//return m_ListOfGameObjects[containerID]->GetNumberOfPickableItems();
	return getGameObjectFromID(containerID)->GetNumberOfPickableItems();
}

CGameObject* CLevel::GetContainerItem(int containerID, int itemID)
{
	return getGameObjectFromID(containerID)->GetPickableItem(itemID);
}

/**
 * \brief Adds triangle selector to the terrain (used on reloading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevel::addTerrainSelector()
{
	ITerrainSceneNode* terrain = (ITerrainSceneNode*) m_SMGR->getSceneNodeFromType(ESNT_TERRAIN);
	scene::ITriangleSelector* selector = m_SMGR->createTerrainTriangleSelector(terrain, 0);
	terrain->setTriangleSelector(selector);
	selector->drop();
}

CGameObject* CLevel::getClickedGameObject(position2d<s32> mousePos)
{
	ISceneNode* node = m_SMGR->getSceneCollisionManager()->getSceneNodeFromScreenCoordinatesBB(mousePos);
	if(node)
	{
		CGameObject* go = getGameObjectFromID(node->getID());
		return go;
	}
	return 0;
}

bool CLevel::isActionItemUnderMousePointer(position2d<s32> mousePos)
{
	ISceneNode* node = m_SMGR->getSceneCollisionManager()->getSceneNodeFromScreenCoordinatesBB(mousePos);
	if(node)
	{
		CGameObject* go = getGameObjectFromID(node->getID());
		if(go)
			if(go->isContainer || go->isPickable || go->isTrigger || go->isNPC || go->isMonster)
				return true;
	}

	return false;
}

bool CLevel::isObjectPickable(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->isPickable;
	}
	printf("ERROR! isObjectPickable: Object with id: %d doesn't exist!",id);
	return false;
}

bool CLevel::isObjectContainer(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->isContainer;
	}
	printf("ERROR! isObjectContainer: Object with id: %d doesn't exist!",id);
	return false;
}

bool CLevel::isObjectTrigger(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->isTrigger;
	}
	printf("ERROR! isObjectTrigger: Object with id: %d doesn't exist!",id);
	return false;
}

bool CLevel::isObjectMonster(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->isMonster;
	}
	printf("ERROR! isObjectMonster: Object with id: %d doesn't exist!",id);
	return false;
}

bool CLevel::isObjectNPC(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->isNPC;
	}
	printf("ERROR! isObjectNPC: Object with id: %d doesn't exist!",id);
	return false;
}

stringw CLevel::GetObjectScript(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->script;
	}
	printf("ERROR! GetObjectScript: Object with id: %d doesn't exist!",id);
	return "";
}

void CLevel::SetObjectState(int id, stringw state)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		go->state = state;
		return;
	}
	printf("ERROR! SetObjectState: Object with id: %d doesn't exist!",id);
}

stringw CLevel::GetObjectState(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->state;
	}
	printf("ERROR! GetObjectState: Object with id: %d doesn't exist!",id);
	return "";
}

vector3df CLevel::GetObjectPosition(int id)
{
	CGameObject* go = getGameObjectFromID(id);
	if(go)
	{
		return go->pos;
	}
	printf("ERROR! GetObjectPosition: Object with id: %d doesn't exist!",id);
	return vector3df(0,10,0);
}
