/** 
 * \file LevelManager.cpp
 * \brief File LevelManager.cpp takes care of all game levels.
 *
 * takes care of saving/loading levels
 * manages list of levels
 * tightly communicates with GUI environment. 
 *
 * \author Petar Bajic, MPE (C) All Rights Reserved, Homepage: www.mystic-peanut.com
 * \date July, 21 2008.
 */

#include "LevelManager.h"
#include "../GameManager.h"
#include "../gui/GameGUI.h"

#define CAMERA_OFFSET vector3df(50,180,50)

/**
 * \brief Standard constructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CLevelManager::CLevelManager()
{
	m_GameManager = NULL;
	m_bShiftPressed = false;
	m_DeletePicked = false;
	m_bHoverOverActionItem = false;
	m_CurrentZoom = vector3df(1.0f,1.0f,1.0f);
	m_NumberOfLoadedLevels = 0;
	m_bMoveToPosition = false;
}

/**
 * \brief Standard destructor.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
CLevelManager::~CLevelManager()
{
}

/**
 * \brief Anything special (not belonging to irrlicht scene) is rendered here
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevelManager::OnRender()
{
	//m_pLevels[m_LevelIndex]->m_SMGR->drawAll();
}

/**
 * \brief Init function stores pointer to CGameManager and creates initial level.
 * \author Petar Bajic
 * \date July, 21 2008.
 */
bool CLevelManager::Init(CGameManager* gameMngr, CGameGUI* gameGUI)
{
	m_GameManager = gameMngr;

	m_LevelIndex = -1;
	
	//what is this for?
	m_GameManager->getDriver()->setTextureCreationFlag(video::ETCF_ALWAYS_32_BIT, true);

	// add irrlicht logo
	m_GameManager->getGUIEnvironment()->addImage(m_GameManager->getDriver()->getTexture("media/irrlichtlogo2.png"), core::position2d<s32>(50,50));

	m_GameGUI = gameGUI;

	for(u32 i=0; i<MAX_NUMBER_OF_LEVELS; i++)
	{
		m_pLevels[i] = 0;
	}

	return true;
}

/**
 * \brief Creates camera in initial position (used for loading map)
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
void CLevelManager::CreateCamera(vector3df position)
{
	m_pCamera = new RTSCamera(m_GameManager->getDevice(),m_pLevels[m_LevelIndex]->m_SMGR->getRootSceneNode(),m_GameManager->getSceneMngr(),-1,100.0f,10.0f,100.0f);
	m_pCamera->setPosition(position + CAMERA_OFFSET);
	m_pCamera->setTarget(position);
}

/**
 * \brief Sets camera position
 * \author Petar Bajic 
 * \date February 10, 2010.
 */
void CLevelManager::SetCameraPos(vector3df position)
{
	m_pCamera->setPosition(position + CAMERA_OFFSET);
	m_pCamera->setTarget(position);
}

stringw CLevelManager::getCurrentMapName()
{
	return m_pLevels[m_LevelIndex]->m_MapName;
}

/**
 * \brief returns initial player position if marked on the map
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
vector3df CLevelManager::GetStartPosition()
{
	return m_pLevels[m_LevelIndex]->GetStartPosition();
}

/**
 * \brief saves level to temp dir. 
 *
 * When player moves from level to level, and happens to return to previous one, 
 * it has to be saved as he left it, even if player haven't explicitly saved the game.
 * That is what temp saved levels are for.
 *
 * \author Petar Bajic 
 * \date January, 24 2010.
 */
void CLevelManager::SavePreviousLevelToTemp()
{
	m_pLevels[m_LevelIndex]->Save(stringc("save/temp/") + m_pLevels[m_LevelIndex]->m_MapName);
}

void CLevelManager::SaveLevels()
{
	int c;
	//FILE *in,*out;
	for(u32 i=0; i<m_NumberOfLoadedLevels; i++)
	{
		stringc tempname = stringc("save/temp/") + m_pLevels[i]->m_MapName;
		stringc savename = stringc("save/") + m_pLevels[i]->m_MapName;
		if(m_GameManager->getFS()->existFile(tempname.c_str()))
		{
			remove(savename.c_str());
			c = rename(tempname.c_str(),savename.c_str());
			if(c) perror(0);
		}
		else
		{
			m_pLevels[i]->Save(stringc("save/") + m_pLevels[i]->m_MapName);
		}
	}
}

/**
 * \brief Determines if this map was already loaded, and returns new index if not.
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
u32 CLevelManager::_GetLevelIndex(stringc map_filename)
{
	u32 index = 0;
	stringc MapName = map_filename.subString(map_filename.findLast('/')+1,map_filename.size());

	for(index=0; index<m_NumberOfLoadedLevels; index++)
	{
		if(m_pLevels[index]->m_MapName == MapName)
		{
			return index;
		}
	}

	return index;
}

/**
 * \brief Manages Map Loading
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CLevelManager::OnLoadMap(stringc map_filename)
{
	//level index is tied to map, if level is to be loaded again, his old index will be waiting for him.
	m_LevelIndex = _GetLevelIndex(map_filename);

	//Create new level if map is loaded for the first time
	if(!m_pLevels[m_LevelIndex])
	{
		m_pLevels[m_LevelIndex] = new CLevel();
		m_pLevels[m_LevelIndex]->Init(m_GameManager->getDevice(),m_GameManager->getFS(),m_GameManager->getDriver(),m_GameManager->getSceneMngr()/*->createNewSceneManager(false)*/);
		m_NumberOfLoadedLevels++;
	}

	//Load map scene from file
	if (m_pLevels[m_LevelIndex]->Load(map_filename))
	{
		//back to working dir
		m_GameManager->backToWorkingDirectory();
		CreateCamera(GetStartPosition());
		return true;
	}

	//Display error to console
	stringw message  = "Map: ";
	message += map_filename;
	message += " can not be loaded!";
	m_GameManager->getGUIEnvironment()->addMessageBox(L"Error Loading Map", message.c_str());

	return false;
}

u32 CLevelManager::FindNPCsOnMap(stringc mapname, stringc NPCNames[])
{
	u32 numNPCs = 0;
	IXMLReader* reader = m_GameManager->getFS()->createXMLReader(mapname.c_str());
	while(reader->read())
	{
		//loop through scene nodes in xml
		switch(reader->getNodeType())
		{
			case io::EXN_ELEMENT:
			{
				if (stringw("attributes") == reader->getNodeName())
				{
					//load node attributes
					IAttributes* attr = m_GameManager->getFS()->createEmptyAttributes(m_GameManager->getDriver());
					attr->read(reader);
					if(attr->getAttributeAsBool("isNPC"))
					{
						stringc meshPath = attr->getAttributeAsString("Mesh");
						NPCNames[numNPCs] = m_GameManager->getRootNameFromPathName(meshPath);
						numNPCs++;
					}
				}
				break;
			}
		}
	}
	return numNPCs;
}

void CLevelManager::RemoveContainerContent(int containerID)
{
	m_pLevels[m_LevelIndex]->RemoveContainerContent(containerID);
}

void CLevelManager::AddContainerItem(int containerID, CGameObject* pick)
{
	m_pLevels[m_LevelIndex]->AddContainerItem(containerID, pick);
}

int CLevelManager::GetContainerNumberOfItems(int containerID)
{
	return m_pLevels[m_LevelIndex]->GetContainerNumberOfItems(containerID);
}

CGameObject* CLevelManager::GetContainerItem(int containerID, int itemID)
{
	return m_pLevels[m_LevelIndex]->GetContainerItem(containerID, itemID);
}

bool CLevelManager::isObjectContainer(int id)
{
	return m_pLevels[m_LevelIndex]->isObjectContainer(id);
}

bool CLevelManager::isObjectPickable(int id)
{
	return m_pLevels[m_LevelIndex]->isObjectPickable(id);
}

bool CLevelManager::isObjectTrigger(int id)
{
	return m_pLevels[m_LevelIndex]->isObjectTrigger(id);
}

bool CLevelManager::isObjectNPC(int id)
{
	return m_pLevels[m_LevelIndex]->isObjectNPC(id);
}

bool CLevelManager::isObjectMonster(int id)
{
	return m_pLevels[m_LevelIndex]->isObjectMonster(id);
}

stringw CLevelManager::GetObjectScript(int id)
{
	return m_pLevels[m_LevelIndex]->GetObjectScript(id);
}

void CLevelManager::SetObjectState(int id, stringw state)
{
	m_pLevels[m_LevelIndex]->SetObjectState(id,state);
}

stringw CLevelManager::GetObjectState(int id)
{
	return m_pLevels[m_LevelIndex]->GetObjectState(id);
}

vector3df CLevelManager::GetObjectPosition(int id)
{
	return m_pLevels[m_LevelIndex]->GetObjectPosition(id);
}

CGameObject* CLevelManager::getGameObjectFromID(int id)
{
	return m_pLevels[m_LevelIndex]->getGameObjectFromID(id);
}

bool CLevelManager::isNodeTerrain(const ISceneNode* node)
{
	CGameObject* go = m_pLevels[m_LevelIndex]->getGameObjectFromID(node->getID());
	if(go)
	{
		if(go->isTerrain)
		{
			return true;
		}
	}

	return false;
}

bool CLevelManager::isNodeActionObject(const ISceneNode* node)
{
	CGameObject* go = m_pLevels[m_LevelIndex]->getGameObjectFromID(node->getID());
	if(go)
	{
		if(go->isContainer || go->isMonster || go->isNPC || go->isPickable || go->isTrigger)
		{
			return true;
		}
	}

	return false;
}

bool CLevelManager::Action(s32 id)
{
	//perform action of game object
	CGameObject* go = m_pLevels[m_LevelIndex]->getGameObjectFromID(id);

	if (go)
	{
		//nonexclusive list of "ifs" allows objects to be manything
		//object can be both container and pickable 
		if (go->isContainer)
		{
			m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONCONTAINERCLICK, go->script, go->id);
			return true;
		}
		if (go->isPickable)
		{
			if(!m_GameManager->getGameGUI()->InventoryFull())
			{
				m_GameManager->getGameGUI()->AddConsoleText(go->name + stringw(L" was picked up."));
				//Move item to inventory automatically
				m_GameManager->getGameGUI()->AddPickableToInventory(go);
				//Plus do the pickup script
				m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONPICK, go->script, go->id);
				//Remove object from the scene
				m_pLevels[m_LevelIndex]->EraseElement(id);
				ISceneNode* pickedNode = m_GameManager->getSceneMngr()->getSceneNodeFromId(id);
				m_pLevels[m_LevelIndex]->m_LevelMetaTriangleSelector->removeTriangleSelector(pickedNode->getTriangleSelector());
				m_pLevels[m_LevelIndex]->m_ObstacleMetaTriangleSelector->removeTriangleSelector(pickedNode->getTriangleSelector());
				pickedNode->remove();
			}
			else
			{
				m_GameManager->getGameGUI()->AddConsoleText(stringw(L"Inventory is full!"));
			}
			//return is needed here because there is a situation when action causes level change (MoveToMap)
			//and rest of the code (isNPC,isTrigger) can not be performed on new level cause it crashes naturally.
			return true;
		}
		if (go->isTrigger)
		{
			if(m_GameManager->getGameGUI()->m_bDraggingPickableItem)
			{
				m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONUSEAGAINST, m_GameManager->getGameGUI()->m_pDraggedPickableItem->script, go->id);
			}
			else
			{
				m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_TRIGGER_ONCLICK, go->script, go->id);
			}
			//return is needed here because there is a situation when action causes level change (MoveToMap)
			//and rest of the code (isNPC check) can not be performed on new level cause it crashes naturally.
			return true;
		}
		if (go->isNPC)
		{
			stringw NPCName = m_GameManager->getRootNameFromPathName(go->mesh);
			NPCName += ".dlg";
			m_GameManager->getGameGUI()->StartNPCDialog(NPCName);
			return true;
		}
	}
	return false;
}

/**
 * \brief Mouse and Keyboard events are handled here. 
 * Every mouse click on game object is caught, interaction with the map is handled:
 * doors lead to another map, containers display their content, NPCs start talking...
 *
 * \author Petar Bajic 
 * \date July, 21 2008.
 */
bool CLevelManager::OnEvent(const SEvent& ovent)
{
	if(ovent.EventType == EET_KEY_INPUT_EVENT)
	{
		if(ovent.KeyInput.Key == KEY_SHIFT)
		{
			m_bShiftPressed = ovent.KeyInput.PressedDown;
		}
	}

	if(ovent.EventType == EET_MOUSE_INPUT_EVENT)
	{
		switch(ovent.MouseInput.Event)
		{
			case EMIE_LMOUSE_PRESSED_DOWN:
				{
					//translate mouse click to game event:
					/*CGameObject* go = m_pLevels[m_LevelIndex]->getClickedGameObject(m_GameManager->getDevice()->getCursorControl()->getPosition());

					if (go)
					{
						//nonexclusive list of "ifs" allows objects to be manything
						//object can be both container and pickable 
						if (go->isContainer)
						{
							m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONCONTAINERCLICK, go->script, go->id);
							return true;
						}
						if (go->isPickable)
						{
							if(!m_GameManager->getGameGUI()->InventoryFull())
							{
								//Move item to inventory automatically
								m_GameManager->getGameGUI()->AddPickableToInventory(go);
								//Plus do the pickup script
								m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONPICK, go->script, go->id);
							}
							else
							{
								m_GameManager->getGameGUI()->AddConsoleText(stringw(L"Inventory is full!"));
							}
							//return is needed here because there is a situation when action causes level change (MoveToMap)
							//and rest of the code (isNPC,isTrigger) can not be performed on new level cause it crashes naturally.
							return true;
						}
						if (go->isTrigger)
						{
							if(m_GameManager->getGameGUI()->m_bDraggingPickableItem)
							{
								m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_ONUSEAGAINST, m_GameManager->getGameGUI()->m_pDraggedPickableItem->script, go->id);
							}
							else
							{
								m_GameManager->m_pScriptEngine->OnEvent(SCRIPT_EVENT_TRIGGER_ONCLICK, go->script, go->id);
							}
							//return is needed here because there is a situation when action causes level change (MoveToMap)
							//and rest of the code (isNPC check) can not be performed on new level cause it crashes naturally.
							return true;
						}
						if (go->isNPC)
						{
							stringw NPCName = m_GameManager->getRootNameFromPathName(go->mesh);
							NPCName += ".dlg";
							m_GameManager->getGameGUI()->StartNPCDialog(NPCName);
							return true;
						}
					}*/
				}
				break;
			case EMIE_LMOUSE_LEFT_UP:
				break;
			case EMIE_MOUSE_MOVED:
				{
					if (!m_GameManager->getGameGUI()->m_bDraggingPickableItem)
					{
						if(m_pLevels[m_LevelIndex]->isActionItemUnderMousePointer(m_GameManager->getDevice()->getCursorControl()->getPosition()))
						{
							m_bHoverOverActionItem = true;
						}
						else
						{
							m_bHoverOverActionItem = false;
						}
					}
					else
					{
						m_bHoverOverActionItem = false;
					}
				}
				break;
			case EMIE_MOUSE_WHEEL:
				{
				}
				break;
			default:
				break;
		}
	}
	
	m_pCamera->OnUvent(ovent);

	return false;
}

void CLevelManager::DropPickableToMap(CGameObject* pick, vector3df position)
{
	m_pLevels[m_LevelIndex]->AddObjectToScene(pick, position);
}

void CLevelManager::MoveCamera(vector3df pos)
{
	vector3df move = pos - m_pCamera->getTarget();
	m_pCamera->setTarget(pos);
	m_pCamera->setPosition(m_pCamera->getPosition()+move);
}
